// CSNode.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "CSNode.h"
#include <Tools/FBuild/FBuildCore/Graph/DirectoryListNode.h>
#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/Flog.h>
#include <Tools/FBuild/FBuildCore/Graph/NodeGraph.h>
#include <Tools/FBuild/FBuildCore/Helpers/CIncludeParser.h>

#include <Core/FileIO/FileIO.h>
#include <Core/FileIO/FileStream.h>
#include <Core/Math/CRC32.h>
#include <Core/Process/Process.h>
#include <Core/Tracing/Tracing.h>
#include <Core/Strings/AStackString.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
CSNode::CSNode( const AString & compilerOutput,
				const Array< Node * > & inputNodes,
				const AString & compiler,
				const AString & compilerArgs,
				const Array< Node * > & extraRefs )
: FileNode( compilerOutput, Node::FLAG_NONE )
, m_StaticDependencies( inputNodes.GetSize() + extraRefs.GetSize(), false )
, m_DynamicDependencies( 0, true )
, m_ExtraRefs( extraRefs )
{
	ASSERT( !inputNodes.IsEmpty() );

	for ( size_t i=0; i<inputNodes.GetSize(); ++i )
	{
		m_StaticDependencies.Append( inputNodes[i] );
	}

	for ( size_t i=0; i<extraRefs.GetSize(); ++i )
	{
		m_StaticDependencies.Append( extraRefs[ i ] );
	}

	// store options we'll need to use when building
	m_CompilerPath = compiler; // TODO:C This should be a node we statically depend on
	m_CompilerArgs = compilerArgs;

	m_Type = CS_NODE;
	m_LastBuildTimeMs = 5000; // higher default than a file node
}

// DESTRUCTOR
//------------------------------------------------------------------------------
CSNode::~CSNode()
{
}

// DoDynamicDependencies
//------------------------------------------------------------------------------
/*virtual*/ bool CSNode::DoDynamicDependencies( bool forceClean )
{
	if ( forceClean == true )
	{
		return true; // includes will be re-built as part of build step
	}

	ASSERT( m_DynamicDependencies.GetSize() == 0 );

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();

	// preallocate a reasonably amount of space
	m_DynamicDependencies.SetCapacity( m_StaticDependencies.GetSize() );

	// convert static deps to dynamic deps
	// (ignore the extra refs here)
	size_t numDeps = m_StaticDependencies.GetSize() - m_ExtraRefs.GetSize();
	for ( size_t i=0; i<numDeps; ++i ) 
	{
		Node * n = m_StaticDependencies[ i ];

		if ( n->IsAFile() )
		{
			m_DynamicDependencies.Append( n );
			continue;
		}

		if ( n->GetType() == Node::DIRECTORY_LIST_NODE )
		{
			// get the list of files
			DirectoryListNode * dln = n->CastTo< DirectoryListNode >();
			const Array< AString > & files = dln->GetFiles();
			m_DynamicDependencies.SetCapacity( m_DynamicDependencies.GetSize() + files.GetSize() );
			for ( Array< AString >::Iter fIt = files.Begin();
					fIt != files.End();
					fIt++ )
			{
				// Create the file node (or find an existing one)
				Node * sn = ng.FindNode( *fIt );
				if ( sn == nullptr )
				{
					sn = ng.CreateFileNode( *fIt );
				}
				else if ( sn->IsAFile() == false )
				{
					FLOG_ERROR( "CSAssembly() .CompilerInputFile '%s' is not a FileNode (type: %s)", n->GetName().Get(), n->GetTypeName() );
					return false;
				}

				m_DynamicDependencies.Append( sn );
			}
			continue;
		}

		FLOG_ERROR( "'%s' is not a supported node type (type: %s)", n->GetName().Get(), n->GetTypeName() );
		return false;
	}

	return true;
}

// DoBuild
//------------------------------------------------------------------------------
/*virtual*/ Node::BuildResult CSNode::DoBuild( Job * UNUSED( job ) )
{
	// Format compiler args string
	AStackString< 4 * KILOBYTE > fullArgs;
	fullArgs = m_CompilerArgs;

	// substitute "function variables" if needed
	AStackString< 4 * KILOBYTE > sourceFiles;
	for ( Array< Node * >::ConstIter it = m_DynamicDependencies.Begin();
		  it != m_DynamicDependencies.End();
		  ++it )
	{
		if ( !sourceFiles.IsEmpty() )
		{
			sourceFiles += ' ';
		}
		sourceFiles += ( *it )->GetName();
	}
	fullArgs.Replace( "%1", sourceFiles.Get() );
	fullArgs.Replace( "%2", m_Name.Get() );

	AStackString<> additionalLinkage;
	for ( size_t i=0; i< m_ExtraRefs.GetSize(); ++i )
	{
		if ( i > 0 )
		{
			additionalLinkage += ',';
		}
		additionalLinkage += m_ExtraRefs[ i ]->GetName().Get();
	}
	fullArgs.Replace( "%3", additionalLinkage.Get() );

	// use the exe launch dir as the working dir
	const char * workingDir = nullptr;

	const char * environment = FBuild::Get().GetEnvironmentString();

	EmitCompilationMessage( fullArgs );

	// spawn the process
	Process p;
	if ( p.Spawn( m_CompilerPath.Get(), fullArgs.Get(),
				  workingDir, environment ) == false )
	{
		FLOG_ERROR( "Failed to spawn process to build '%s'", GetName().Get() );
		return NODE_RESULT_FAILED;
	}

	// capture all of the stdout and stderr
	AutoPtr< char > memOut;
	AutoPtr< char > memErr;
	uint32_t memOutSize = 0;
	uint32_t memErrSize = 0;
	p.ReadAllData( memOut, &memOutSize, memErr, &memErrSize );

	// Get result
	ASSERT( !p.IsRunning() );
	int result = p.WaitForExit();
	bool ok = ( result == 0 );

	if ( !ok )
	{
		// something went wrong, print details
		Node::DumpOutput( memOut.Get(), memOutSize );
		Node::DumpOutput( memErr.Get(), memErrSize );
		goto failed;
	}

	if ( !FileIO::FileExists( m_Name.Get() ) )
	{
		FLOG_ERROR( "Object missing despite success for '%s'", GetName().Get() );
		return NODE_RESULT_FAILED;
	}

	// record new file time
	m_TimeStamp = FileIO::GetFileLastWriteTime( m_Name );

	return NODE_RESULT_OK;

failed:
	FLOG_ERROR( "Failed to build Object (error %i) '%s'", result, GetName().Get() );

	return NODE_RESULT_FAILED;
}

// GetStaticDependencies
//------------------------------------------------------------------------------
/*virtual*/ const Array< Node * > & CSNode::GetStaticDependencies() const
{
	return m_StaticDependencies;
}

// GetDynamicDependencies
//------------------------------------------------------------------------------
/*virtual*/ const Array< Node * > & CSNode::GetDynamicDependencies() const
{
	return m_DynamicDependencies;
}

// DetermineNeedToBuild
//------------------------------------------------------------------------------
bool CSNode::DetermineNeedToBuild( bool forceClean )
{
	// 
	uint64_t lastWriteTime = FileIO::GetFileLastWriteTime( m_Name );
	if ( lastWriteTime == 0 )
	{
		// output file is missing, so it must be built
		return true;
	}

	if ( GetTimeStamp() == 0 )
	{
		// file exists, be we don't know about it
		// (built by someone else, or we're doing a clean build)
		return true;
	}

	if ( lastWriteTime != GetTimeStamp() )
	{
		// file exists and we've built before, but they are out of sync
		// (probably modified or built by an external process)
		return true;
	}

	FileNode * staticDep = m_StaticDependencies[ 0 ]->CastTo< FileNode >();
	ASSERT( staticDep->GetTimeStamp() != 0 );  // Should never attempt build if a static dep is missing

	// is our output older than the source file?
	// also handles the case where 
	if ( m_TimeStamp < staticDep->GetTimeStamp() )
	{
		return true;
	}

	if ( forceClean )
	{
		return true;
	}

	// we have a list of includes - are any out of date or missing?
	// TODO:A does this make sense for c# nodes?
	for ( Array< Node * >::ConstIter it = m_DynamicDependencies.Begin();
			it != m_DynamicDependencies.End();
			it++ )
	{
		FileNode * fn = ( *it )->CastTo< FileNode >();
		if ( fn->GetTimeStamp() == 0 )
		{
			// file missing
			return true;
		}

		if ( m_TimeStamp < fn->GetTimeStamp() )
		{
			// file changed
			return true;
		}
	}

	// nothing needs building
	return false;
}

// Load
//------------------------------------------------------------------------------
/*static*/ Node * CSNode::Load( IOStream & stream, bool remote )
{
	NODE_LOAD( AStackString<>,	name );
	NODE_LOAD( uint64_t,		timeStamp );
	NODE_LOAD_DEPS( 2,			staticDeps );
	NODE_LOAD( AStackString<>,	compilerPath );
	NODE_LOAD( AStackString<>,	compilerArgs );
	NODE_LOAD_DEPS( 0,			extraRefs );

	ASSERT( staticDeps.GetSize() >= 1 );

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();
	Node * on = ng.CreateCSNode( name, staticDeps, compilerPath, compilerArgs, extraRefs );
	CSNode * csNode = on->CastTo< CSNode >();
	csNode->m_TimeStamp = timeStamp;
	return csNode;
}

// Save
//------------------------------------------------------------------------------
/*virtual*/ bool CSNode::Save( IOStream & stream ) const
{
	NODE_SAVE( m_Name );
	NODE_SAVE( m_TimeStamp );

	// Only save the original static deps here (remove the extra refs)
	size_t numBaseDeps = m_StaticDependencies.GetSize() - m_ExtraRefs.GetSize();
	Array< Node * > staticDeps( numBaseDeps, false );
	for ( size_t i=0; i<numBaseDeps; ++i )
	{
		staticDeps.Append( m_StaticDependencies[ i ] );
	}
	NODE_SAVE_DEPS( staticDeps );

	NODE_SAVE( m_CompilerPath );
	NODE_SAVE( m_CompilerArgs );
	NODE_SAVE_DEPS( m_ExtraRefs );
	return true;
}

// EmitCompilationMessage
//------------------------------------------------------------------------------
void CSNode::EmitCompilationMessage( const AString & fullArgs ) const
{
	// print basic or detailed output, depending on options
	// we combine everything into one string to ensure it is contiguous in
	// the output
	AStackString<> output;
	output += "C#: ";
	//FileNode * sourceFile = m_StaticDependencies[ 0 ]->CastTo< FileNode >();
	//output += sourceFile->GetName().FindLast( '\\' ) + 1;
	//output += " (";
	output += GetName();
	//output += ")\n";
	output += '\n';
	if ( FLog::ShowInfo() )
	{
		output += m_CompilerPath;
		output += ' ';
		output += fullArgs;
		output += '\n';
	}
	FLOG_BUILD( "%s", output.Get() );
}

//------------------------------------------------------------------------------
