// FLog
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "FLog.h"

#include <Tools/FBuild/FBuildCore/WorkerPool/WorkerThread.h>
#include <Tools/FBuild/FBuildCore/FBuild.h>

#include <Core/Env/Types.h>
#include <Core/Tracing/Tracing.h>

#include <stdio.h>
#include <stdarg.h>
#ifdef DEBUG
	#include <windows.h> // for OutputDebugStringA
#endif
// Static Data
//------------------------------------------------------------------------------
/*static*/ bool FLog::s_ShowInfo = false;
/*static*/ bool FLog::s_ShowErrors = true;
/*static*/ bool FLog::s_ShowProgress = false;
/*static*/ AStackString< 64 > FLog::m_ProgressText;

// Info
//------------------------------------------------------------------------------
//#pragma warning( push )
//#pragma warning( disable : 28302 ) // static analysis warns about vsnprintf_s "For C++ reference-parameter '_Param_(1)', an extra _Deref_ operator was found on SAL_null."
/*static*/ void FLog::Info( const char * formatString, ... )
{
	const size_t BUFFER_SIZE( MAX_MESSAGE_LENGTH );
	char buffer[ BUFFER_SIZE ];

	va_list args;
	va_start(args, formatString);
	vsnprintf_s( buffer, BUFFER_SIZE, _TRUNCATE, formatString, args );
	va_end( args );

	Output( "Info:", buffer );
}
//#pragma warning( pop ) // for vsnprintf_s

// Build
//------------------------------------------------------------------------------
/*static*/ void FLog::Build( const char * formatString, ... )
{
	const size_t BUFFER_SIZE( MAX_MESSAGE_LENGTH );
	char buffer[ BUFFER_SIZE ];

	va_list args;
	va_start(args, formatString);
	vsnprintf_s( buffer, BUFFER_SIZE, _TRUNCATE, formatString, args );
	va_end( args );

	Output( nullptr, buffer );
}

// Warning
//------------------------------------------------------------------------------
/*static*/ void FLog::Warning( const char * formatString, ... )
{
	const size_t BUFFER_SIZE( MAX_MESSAGE_LENGTH );
	char buffer[ BUFFER_SIZE ];

	va_list args;
	va_start(args, formatString);
	vsnprintf_s( buffer, BUFFER_SIZE, _TRUNCATE, formatString, args );
	va_end( args );

	Output( "Warning:", buffer );
}

// Error
//------------------------------------------------------------------------------
/*static*/ void FLog::Error( const char * formatString, ... )
{
	// we prevent output here, rather than where the macros is inserted
	// as an error being output is not the normal code path, and a check
	// before calling this function would bloat the code
	if ( FLog::ShowErrors() == false )
	{
		return;
	}

	const size_t BUFFER_SIZE( MAX_MESSAGE_LENGTH );
	char buffer[ BUFFER_SIZE ];

	va_list args;
	va_start(args, formatString);
	vsnprintf_s( buffer, BUFFER_SIZE, _TRUNCATE, formatString, args );
	va_end( args );

	Output( "Error:", buffer );
}

// ErrorDirect
//------------------------------------------------------------------------------
/*static*/ void FLog::ErrorDirect( const char * message )
{
	if ( FLog::ShowErrors() == false )
	{
		return;
	}

	Tracing::Output( message );
}

// Output - write to stdout and debugger
//------------------------------------------------------------------------------
/*static*/ void FLog::Output( const char * type, const char * message )
{
	if( type == nullptr )
	{
		OUTPUT( "%s", message );
		return;
	}

	AStackString< 1024 > buffer( message );
	if ( buffer.IsEmpty() )
	{
		return;
	}
	if ( buffer[ buffer.GetLength() - 1 ] != '\n' )
	{
		buffer += '\n';
	}

	OUTPUT( "%s", buffer.Get() );
}

// StartBuild
//------------------------------------------------------------------------------
/*static*/ void FLog::StartBuild()
{
//	if ( s_ShowProgress )
	{
		Tracing::SetCallbackOutput( &TracingOutputCallback );
	}
}

// StopBuild
//------------------------------------------------------------------------------
/*static*/ void FLog::StopBuild()
{
	if ( s_ShowProgress )
	{
		Tracing::SetCallbackOutput( nullptr );
		puts( "\r                                                               \r" );
		#ifdef DEBUG
			AStackString<> dbgBuffer;
			dbgBuffer.Format( "\n%s\n", m_ProgressText.Get() );
			OutputDebugStringA( dbgBuffer.Get() );
		#endif
		m_ProgressText.Clear();
	}
}

// OutputProgress
//------------------------------------------------------------------------------
/*static*/ void FLog::OutputProgress( float time, 
									  float percentage,
									  uint32_t numJobs, 
									  uint32_t numJobsActive, 
									  uint32_t numJobsDist, 
									  uint32_t numJobsDistActive )
{
	ASSERT( s_ShowProgress );

	m_ProgressText.Format( "%2.1f %%\t[", percentage );

	// 20 column output (100/20 = 5% per char)
	uint32_t numStarsDone = (uint32_t)( percentage * 20.0f / 100.0f ); // 20 columns
	for ( uint32_t i=0; i<numStarsDone; ++i ) 
	{
		m_ProgressText += '*';
	}
	uint32_t numStarsRemaining = ( 20 - numStarsDone );
	for ( uint32_t i=0; i<numStarsRemaining; ++i )
	{
		m_ProgressText += '-';
	}
	m_ProgressText += "] ";
	static int animIndex = 0;
	static char anim[] = { '\\', '|', '/', '-', '\\', '|', '/', '-' };
	m_ProgressText += anim[ ( animIndex++ ) % 8 ]; 
	m_ProgressText += ' ';

	uint32_t timeTakenMinutes = uint32_t( time / 60.0f );
	uint32_t timeTakenSeconds = (uint32_t)time - ( timeTakenMinutes * 60 );
	if ( timeTakenMinutes > 0 )
	{
		AStackString< 16 > bufferM;
		bufferM.Format( " %um", timeTakenMinutes );
		m_ProgressText += bufferM.Get();
	}
	AStackString< 32 > bufferS;
	if ( FBuild::Get().GetOptions().m_AllowDistributed )
	{
		bufferS.Format( " %02us ( %u/%u + %u/%u )", timeTakenSeconds, 
													numJobsActive, 
													numJobsActive + numJobs, 
													numJobsDistActive, 
													numJobsDistActive + numJobsDist );
	}
	else
	{
		bufferS.Format( " %02us ( %u/%u )", timeTakenSeconds, 
											numJobsActive, 
											numJobsActive + numJobs );
	}
	m_ProgressText += bufferS.Get();

	if ( FBuild::Get().GetStopBuild() )
	{
		m_ProgressText.Format( "BUILD ABORTED - STOPPING %c", anim[ ( animIndex++ ) % 8 ] );
	}

	TracingOutputCallback( "" );
}

// TracingOutputCallback
//------------------------------------------------------------------------------
/*static*/ bool FLog::TracingOutputCallback( const char * message )
{
	uint32_t threadIndex = WorkerThread::GetThreadIndex();

	AStackString< 2048 > tmp;

	if ( s_ShowProgress )
	{
		// clear previous progress message
		tmp += "\r                                                               \r";
	}

	// print output and then progress
	if ( threadIndex > 0 )
	{
		char buffer[ 8 ];
		sprintf_s( buffer, 8, "%u>", threadIndex );
		tmp += buffer;
		tmp += message;
		tmp += m_ProgressText;
		#ifdef DEBUG
			AStackString<> dbgBuffer;
			dbgBuffer.Format( "%u> %s", threadIndex, message );
			OutputDebugStringA( dbgBuffer.Get() );
		#endif
	}
	else
	{
		tmp += message;
		tmp += m_ProgressText;
		#ifdef DEBUG
			OutputDebugStringA( message );
		#endif
	}

	if ( s_ShowProgress )
	{
		#ifdef DEBUG
			OutputDebugStringA( m_ProgressText.Get() );
		#endif
	}

	fputs( tmp.Get(), stdout );

	return false; // tell tracing not to output it again
}

//------------------------------------------------------------------------------
