// FBuild - the main application
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "FBuild.h"

#include "FLog.h"
#include "BFF/BFFParser.h"
#include "BFF/Functions/Function.h"
#include "Graph/Node.h"
#include "Graph/NodeGraph.h"
#include "Protocol/Client.h"
#include "Protocol/Protocol.h"
#include "WorkerPool/JobQueue.h"
#include "WorkerPool/WorkerThread.h"

#include <Core/Env/Assert.h>
#include <Core/Env/Env.h>
#include <Core/Env/Types.h>
#include <Core/FileIO/FileIO.h>
#include <Core/FileIO/FileStream.h>
#include <Core/Strings/AStackString.h>
#include <Core/Tracing/Tracing.h>

#include <stdio.h>
#include <time.h>
#include <windows.h>

//#define DEBUG_CRT_MEMORY_USAGE // Uncomment this for (very slow) detailed mem checks
#ifdef DEBUG_CRT_MEMORY_USAGE
	#include <crtdbg.h>
#endif

// Static
//------------------------------------------------------------------------------
/*static*/ bool FBuild::s_StopBuild( false );

// CONSTRUCTOR - FBuildOptions
//------------------------------------------------------------------------------
FBuildOptions::FBuildOptions()
: m_ForceCleanBuild( false )
, m_UseCacheRead( false )
, m_UseCacheWrite( false )
, m_ShowInfo( false )
, m_ShowProgress( false )
, m_ShowErrors( true )
, m_AllowDistributed( false )
, m_ShowSummary( false )
{
#ifdef DEBUG
	//m_ShowInfo = true; // uncomment this to enable spam when debugging
#endif

	// Default to NUMBER_OF_PROCESSORS
	m_NumWorkerThreads = Env::GetNumProcessors();
}

// FBuildOptions::SetWorkingDir
//------------------------------------------------------------------------------
void FBuildOptions::SetWorkingDir( const AString & path )
{
	m_WorkingDir = path;
	if ( m_WorkingDir.IsEmpty() )
	{
		return;
	}

	// clean path

	// slashed must face the right way
	m_WorkingDir.Replace( '/', '\\' );

	// eliminate redundant slash pairs
	while ( m_WorkingDir.Replace( "\\\\", "\\" ) );

	// no trailing slash
	if ( m_WorkingDir.EndsWith( '\\' ) )
	{
		m_WorkingDir.SetLength( m_WorkingDir.GetLength() - 1 );
	}
}

// CONSTRUCTOR - FBuild
//------------------------------------------------------------------------------
FBuild::FBuild( const FBuildOptions & options )
	: m_LastUpdateTime( 0.0f )
	, m_Client( nullptr )
	, m_WorkerList( 0, true )
	, m_EnvironmentString( nullptr )
	, m_EnvironmentStringSize( 0 )
{
	#ifdef DEBUG_CRT_MEMORY_USAGE
		_CrtSetDbgFlag( _CRTDBG_ALLOC_MEM_DF | 
						_CRTDBG_CHECK_ALWAYS_DF | //_CRTDBG_CHECK_EVERY_16_DF |
						_CRTDBG_CHECK_CRT_DF | 
						_CRTDBG_DELAY_FREE_MEM_DF | 
						_CRTDBG_LEAK_CHECK_DF );
	#endif
	m_DependencyGraph = new NodeGraph();

	// store all user provided options
	m_Options = options;

	// if working dir is not specified, use the current dir
	if ( options.GetWorkingDir().IsEmpty() )
	{
		AStackString<> workingDir;
		VERIFY( FileIO::GetCurrentDir( workingDir ) );
		m_Options.SetWorkingDir( workingDir );
	}

	// poke options where required
	FLog::SetShowInfo( m_Options.m_ShowInfo );
	FLog::SetShowErrors( m_Options.m_ShowErrors );
	FLog::SetShowProgress( m_Options.m_ShowProgress );

	Function::Create();
}

// DESTRUCTOR
//------------------------------------------------------------------------------
FBuild::~FBuild()
{
	Function::Destroy();

	delete m_DependencyGraph;
	delete m_Client;
	delete [] m_EnvironmentString;

	// restore the old working dir to restore
	if ( ( m_Options.GetWorkingDir().IsEmpty() == false ) && 
		 ( m_OldWorkingDir.IsEmpty() == false ) )
	{
		// set the new woring dir
		if ( SetCurrentDirectory( m_OldWorkingDir.Get() ) == FALSE )
		{
			FLOG_ERROR( "Failed to restore working dir: '%s' (error: %u)", m_OldWorkingDir.Get(), GetLastError() );
		}
	}
}

// Initialize
//------------------------------------------------------------------------------
bool FBuild::Initialize( const char * nodeGraphDBFile )
{
	// handle working dir
	if ( m_Options.GetWorkingDir().IsEmpty() == false )
	{
		// store the old working dir to restore
		char oldWorkingDir[ MAX_PATH ];
		if( 0 == GetCurrentDirectory( MAX_PATH, oldWorkingDir ) )
		{
			FLOG_ERROR( "Failed to get working dir (error: %u)", GetLastError() );
			return false;
		}
		m_OldWorkingDir = oldWorkingDir;

		// set the new woring dir
		if ( SetCurrentDirectory( m_Options.GetWorkingDir().Get() ) == FALSE )
		{
			FLOG_ERROR( "Failed to set working dir: '%s' (error: %u)", m_Options.GetWorkingDir().Get(), GetLastError() );
			return false;
		}
	}

	const char * bffFile = m_Options.m_ConfigFile.IsEmpty() ? GetDefaultBFFFileName()
														    : m_Options.m_ConfigFile.Get();
	if ( m_DependencyGraph->Initialize( bffFile, nodeGraphDBFile ) == false )
	{
		return false;
	}

	// if the cache is enabled, make sure the path is set and accessible
	if ( m_Options.m_UseCacheRead || m_Options.m_UseCacheWrite )
	{
		if ( m_CachePath.IsEmpty() )
		{
			FLOG_WARN( "CachePath not set - Caching disabled" );
			m_Options.m_UseCacheRead = false;
			m_Options.m_UseCacheWrite = false;
		}
		else if ( FileIO::EnsurePathExists( m_CachePath ) == false )
		{
			FLOG_WARN( "Cache inaccessible - Caching disabled (%s)", m_CachePath.Get() );
			m_Options.m_UseCacheRead = false;
			m_Options.m_UseCacheWrite = false;
		}
	}

	//
	// create the connection management system if we might need it
	if ( m_Options.m_AllowDistributed )
	{
		if ( m_WorkerList.IsEmpty() )
		{
			FLOG_WARN( "No workers configured - Distributed compilation disabled" );
			m_Options.m_AllowDistributed = false;
		}
		else
		{
			m_Client = new Client( m_WorkerList );
		}
	}

	return true;
}

// Build
//------------------------------------------------------------------------------
bool FBuild::Build( const AString & target )
{
	ASSERT( !target.IsEmpty() );

	// get the node being requested (search for exact match, to find aliases etc first)
	Node * node = m_DependencyGraph->FindNodeInternal( target );
	if ( node == nullptr )
	{
		// failed to find the node, try looking for a fully pathed equivalent
		node = m_DependencyGraph->FindNode( target );
	}

	if ( node == nullptr )
	{
		FLOG_ERROR( "Unknown build target '%s'", target.Get() );
		return false;
	}

	// build it!
	return Build( node );
}

// SaveDependencyGraph
//------------------------------------------------------------------------------
bool FBuild::SaveDependencyGraph( const char * nodeGraphDBFile ) const
{
	nodeGraphDBFile = nodeGraphDBFile ? nodeGraphDBFile : GetDependencyGraphFileName();

	FLOG_INFO( "Saving DepGraph '%s'", nodeGraphDBFile );

	// try to open the file
	FileStream fileStream;
	if ( fileStream.Open( nodeGraphDBFile, FileStream::WRITE_ONLY ) == false )
	{
		// failing to open the dep graph for saving is a serious problem
		FLOG_ERROR( "Failed to open DepGraph for saving '%s'", nodeGraphDBFile );
		return false;
	}

	Timer t;
	if ( m_DependencyGraph->Save( fileStream ) == false )
	{
		FLOG_ERROR( "Saving DepGraph FAILED!" );
		return false;
	}

	FLOG_INFO( "Saving DepGraph Complete in %2.2fs", t.GetElapsed() );

	return true;
}

// Build
//------------------------------------------------------------------------------
bool FBuild::Build( Node * nodeToBuild )
{
	ASSERT( nodeToBuild );

	// create worker threads
	m_JobQueue = new JobQueue( m_Options.m_NumWorkerThreads );

	m_Timer.Start();
	m_LastUpdateTime = 0.0f;
	FLog::StartBuild();

	// keep doing build passes until completed/failed
	while ( ( nodeToBuild->GetState() != Node::UP_TO_DATE ) &&
			( nodeToBuild->GetState() != Node::FAILED ) )
	{	
		if ( s_StopBuild == true )
		{
			break;
		}

		// do a sweep of the graph
		m_DependencyGraph->DoBuildPass( nodeToBuild );

		// update progress
		UpdateBuildStatus( nodeToBuild, false );

		Sleep( 1 );

		JobQueue::Get().FinalizeCompletedJobs();
	}

	// free the network distribution system (if there is one)
	delete m_Client;
	m_Client = nullptr;

	// wait for workers to exit.  Can still be building even though we've failed:
	//  - only 1 failed node propagating up to root while others are not yet complete
	//  - aborted build, so workers can be incomplete
	m_JobQueue->SignalStopWorkers();
	while ( m_JobQueue->HaveWorkersStopped() == false )
	{
		if ( m_Options.m_ShowProgress )
		{
			FLog::OutputProgress( 0.0f, 0.0f, 0, 0, 0, 0 ); // will print aborting msg, so real values are not needed
		}
		Sleep( 1 );
	}

	delete m_JobQueue;
	m_JobQueue = nullptr;

	// force one more update to show completion time
	UpdateBuildStatus( nodeToBuild, true );

	FLog::StopBuild();

	bool buildOK = ( nodeToBuild->GetState() == Node::UP_TO_DATE );

	float timeTaken = m_Timer.GetElapsed();
	m_BuildStats.m_TotalBuildTime = timeTaken;

	m_BuildStats.GatherPostBuildStatistics( nodeToBuild );

	if ( m_Options.m_ShowSummary )
	{
		m_BuildStats.OutputSummary();
	}

	// final line of output - status of build
	uint32_t minutes = uint32_t( timeTaken / 60.0f );
	timeTaken -= ( minutes * 60.0f );
	float seconds = timeTaken;
	const char * msgString = buildOK ? "Build complete for '%s' in %um %05.2fs\n" 
									 : "Build FAILED for '%s' in %um %05.2fs\n";
	FLOG_BUILD( msgString, nodeToBuild->GetName().Get(), minutes, seconds );
	return buildOK;
}

// SetEnvironmentString
//------------------------------------------------------------------------------
void FBuild::SetEnvironmentString( const char * envString, uint32_t size )
{
	delete [] m_EnvironmentString;
	m_EnvironmentString = new char[ size + 1 ];
	m_EnvironmentStringSize = size;
	AString::Copy( envString, m_EnvironmentString, size );
}

// UpdateBuildStatus
//------------------------------------------------------------------------------
void FBuild::UpdateBuildStatus( const Node * node, bool forceUpdate )
{
	if ( ( FBuild::Get().GetOptions().m_ShowProgress == false ) &&
		 ( forceUpdate == false ) )
	{
		return;
	}

	float timeNow = m_Timer.GetElapsed();

	bool doUpdate = false;
	if ( ( node->GetState() == Node::FAILED ) ||
		 ( node->GetState() == Node::UP_TO_DATE ) )
	{
		doUpdate = true; // force update for build completion
		m_LastUpdateTime = timeNow;
	}
	else if ( m_LastUpdateTime == 0.0f )
	{
		doUpdate = true; // force update for first call
		m_LastUpdateTime = timeNow;
	}
	else
	{
		if ( timeNow > ( m_LastUpdateTime + 1.0f ) )
		{
			doUpdate = true;
			m_LastUpdateTime = timeNow;
		}
	}

	if ( ( doUpdate == false ) && ( forceUpdate == false ) )
	{
		return;
	}

	// re-calculate build stats
	FBuildStats & bs = m_BuildStats;
	bs.m_NodeTimeProgressms = 0;
	bs.m_NodeTimeTotalms = 0;
	bs.m_NumNodesInTarget = 0;
	m_DependencyGraph->UpdateBuildStatusRecurse( node, bs.m_NodeTimeProgressms, bs.m_NodeTimeTotalms, bs.m_NumNodesInTarget );

	// don't output if disabled (even if an update was forced)
	if ( FBuild::Get().GetOptions().m_ShowProgress == false )
	{
		return;
	}

	// get nodes counts;
	uint32_t numJobs = 0;
	uint32_t numJobsActive = 0;
	uint32_t numJobsDist = 0;
	uint32_t numJobsDistActive = 0;
	if ( JobQueue::IsValid() )
	{
		JobQueue::Get().GetJobStats( numJobs, numJobsActive, numJobsDist, numJobsDistActive );
	}

	float doneRatio = (float)( (double)bs.m_NodeTimeProgressms / (double)bs.m_NodeTimeTotalms );
	FLog::OutputProgress( timeNow, ( doneRatio * 100.f ), numJobs, numJobsActive, numJobsDist, numJobsDistActive );
}

// GetDependencyGraphFileName
//------------------------------------------------------------------------------
/*static*/ const char * FBuild::GetDependencyGraphFileName()
{
	return "fbuild.fdb";
}

// GetDefaultBFFFileName
//------------------------------------------------------------------------------
/*static*/ const char * FBuild::GetDefaultBFFFileName()
{
	return "fbuild.bff";
}

// SetCachePath
//------------------------------------------------------------------------------
void FBuild::SetCachePath( const AString & path )
{ 
	m_CachePath = path;
	if ( m_CachePath.EndsWith( '\\' ) == false )
	{
		m_CachePath += '\\';
	}
}

// GetCacheFileName
//------------------------------------------------------------------------------
void FBuild::GetCacheFileName( uint32_t keyA, uint32_t keyB,
						   AString & path ) const
{
	// cache version - bump if cache format is changed
	static const int cacheVersion( 2 );

	const AString & cacheRoot = GetCachePath();
	ASSERT( cacheRoot.IsEmpty() == false );

	// cache is split into sub-folders so there aren't so many files in one dir
	// in order to avoid SMB network filesystem slowdowns
	uint8_t byte1 = uint8_t( keyA & 0x000000FF );
	uint8_t byte2 = uint8_t( ( keyA & 0x0000FF00 ) >> 8 );

	// format example: N:\\fbuild.cache\\32\\DE\\2377DE32_FED872A1.2 
	path.Format( "%s\\%02X\\%02X\\%08X_%08X.%u", cacheRoot.Get(), byte1, byte2, keyA, keyB, cacheVersion );
}

//------------------------------------------------------------------------------
