// FunctionUnity
//------------------------------------------------------------------------------

#include "FunctionUnity.h"
#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/BFF/BFFIterator.h>
#include <Tools/FBuild/FBuildCore/BFF/BFFStackFrame.h>
#include <Tools/FBuild/FBuildCore/BFF/BFFVariable.h>
#include <Tools/FBuild/FBuildCore/Graph/NodeGraph.h>
#include <Tools/FBuild/FBuildCore/Graph/DirectoryListNode.h>

#include <Core/Strings/AStackString.h>

// UnityNode
//------------------------------------------------------------------------------
class UnityNode;

// CONSTRUCTOR
//------------------------------------------------------------------------------
FunctionUnity::FunctionUnity()
: Function( "Unity" )
{
}

// AcceptsHeader
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionUnity::AcceptsHeader() const
{
	return true;
}

// NeedsHeader
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionUnity::NeedsHeader() const
{
	return true;
}

// Commit
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionUnity::Commit( const BFFIterator & funcStartIter ) const
{
	const BFFVariable * inputPathExcludeV;
	const BFFVariable * inputPatternV;
	bool inputPathRecurse;
	const BFFVariable * outputPathV;
	const BFFVariable * outputPatternV;
	if ( !GetString( funcStartIter, inputPathExcludeV, ".UnityInputExcludePath" ) ||
		 !GetString( funcStartIter, inputPatternV,	".UnityInputPattern" ) ||
		 !GetBool( funcStartIter, inputPathRecurse,".UnityInputPathRecurse", true ) ||
		 !GetString( funcStartIter, outputPathV,	".UnityOutputPath", true ) ||
		 !GetString( funcStartIter, outputPatternV,	".UnityOutputPattern" ) )
	{
		return false;
	}

	// UnityInputPath must be provided
	const BFFVariable * inputPathV =  BFFStackFrame::GetVar( ".UnityInputPath" );
	if ( inputPathV == nullptr )
	{
		Error::Error_1101_MissingProperty( funcStartIter, this, AStackString<>( ".UnityInputPath" ) );
		return false;
	}

	// UnityInputPath can be string or array of strings
	Array< AString > paths;
	if ( inputPathV->IsString() )
	{
		paths.Append( inputPathV->GetValue() );
	}
	else if ( inputPathV->IsArray() )
	{
		paths = inputPathV->GetArray();
	}
	else
	{
		Error::Error_1050_PropertyMustBeOfType( funcStartIter, this, "UnityInputPath", inputPathV->GetType(), BFFVariable::VAR_STRING, BFFVariable::VAR_ARRAY );
		return false;
	}

	// num files is optional, but must be an integer
	const BFFVariable * numFilesV = BFFStackFrame::GetVar( ".UnityNumFiles" );
	if ( numFilesV )
	{
		if ( numFilesV->IsInt() == false )
		{
			Error::Error_1050_PropertyMustBeOfType( funcStartIter, this, "UnityNumFiles", numFilesV->GetType(), BFFVariable::VAR_INT );
			return false;
		}
		int numFiles = numFilesV->GetInt();
		if ( ( numFiles < 1 ) || ( numFiles > 999 ) )
		{
			Error::Error_1054_IntegerOutOfRange( funcStartIter, this, "UnityNumFiles", 1, 999 );
			return false;
		}
	}


	// clean exclude path
	AStackString<> excludePath;
	if ( inputPathExcludeV )
	{
		NodeGraph::CleanPath( inputPathExcludeV->GetValue(), excludePath );
		if ( excludePath.EndsWith( '\\' ) == false )
		{
			excludePath += '\\';
		}
	}

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();

	const AString & inputPattern = inputPatternV ? inputPatternV->GetValue() : (const AString &)AStackString<>( "*.cpp" );
	const AString & outputPattern = outputPatternV ? outputPatternV->GetValue() : (const AString &)AStackString<>( "Unity*.cpp" );

	// create all of the DirectoryListNodes we need
	Array< DirectoryListNode * > dirNodes( paths.GetSize() );
	auto end = paths.End();
	for ( auto it = paths.Begin(); it != end; ++it )
	{
		const AString & path = *it;

		// clean path
		AStackString< 512 > fullPath;
		NodeGraph::CleanPath( path, fullPath );
		if ( fullPath.EndsWith( '\\' ) == false )
		{
			fullPath += '\\';
		}

		// get node for the dir we depend on
		AStackString<> dirNodeName;
		DirectoryListNode::FormatName( fullPath, 
									   inputPattern, 
									   inputPathRecurse, 
									   excludePath,
									   dirNodeName );
		Node * dirNode = ng.FindNode( dirNodeName );
		if ( dirNode == nullptr )
		{
			dirNode = ng.CreateDirectoryListNode( dirNodeName,
												  fullPath,
												  inputPattern,
												  inputPathRecurse,
												  excludePath );
		}
		else if ( dirNode->GetType() != Node::DIRECTORY_LIST_NODE )
		{
			Error::Error_1102_UnexpectedType( funcStartIter, this, "UnityInputPath", dirNode->GetName(), dirNode->GetType(), Node::DIRECTORY_LIST_NODE );
			return false;
		}
		dirNodes.Append( dirNode->CastTo< DirectoryListNode >() );
	}

	// uses precompiled header?
	AStackString< 512 > precompiledHeader;
	const BFFVariable * pchV = nullptr;
	if ( !GetString( funcStartIter, pchV, ".UnityPCH" ) )
	{
		return false;
	}
	if ( pchV )
	{
		// Note: we deliberately don't clean the path to the PCH
		// as we don't depend on it, it doesn't need changing
		// and we don't want to mess with the users intent when possible
		precompiledHeader = pchV->GetValue();
	}

	// determine files to exclude
	Array< AString > filesToExclude( 0, true );

	// user excluded files?
	const BFFVariable * excludedFilesV = BFFStackFrame::GetVar( ".UnityInputExcludedFiles" );
	if ( excludedFilesV )
	{
		if ( excludedFilesV->IsString() )
		{
			filesToExclude.SetCapacity( 2 ); // enough for this string + pch file
			filesToExclude.Append( excludedFilesV->GetValue() );
		}
		else if ( excludedFilesV->IsArray() )
		{
			filesToExclude.SetCapacity( excludedFilesV->GetArray().GetSize() + 1 ); // enough for this string + pch file
			auto it = excludedFilesV->GetArray().Begin();
			auto end = excludedFilesV->GetArray().End();
			for ( ; it != end; ++it )
			{
				filesToExclude.Append( *it );
			}
		}
		else
		{
			Error::Error_1050_PropertyMustBeOfType( funcStartIter, this, "UnityInputExcludedFiles", excludedFilesV->GetType(), BFFVariable::VAR_STRING, BFFVariable::VAR_ARRAY );
			return false;
		}
	}

	// automatically exclude the associated CPP file for a PCH (if there is one)
	if ( precompiledHeader.EndsWithI( ".h" ) )
	{
		AStackString<> pchCPP( precompiledHeader.Get(), 
							   precompiledHeader.Get() + precompiledHeader.GetLength() - 2 );
		pchCPP += ".cpp";
		filesToExclude.Append( pchCPP );
	}

	// parsing logic should guarantee we have a string for our name
	ASSERT( m_AliasForFunction.IsEmpty() == false );

	AStackString< 512 > fullOutputPath;
	NodeGraph::CleanPath( outputPathV->GetValue(), fullOutputPath );
	if ( fullOutputPath.EndsWith( '\\' ) == false )
	{
		fullOutputPath += '\\';
	}

	// Check for existing node
	if ( ng.FindNode( m_AliasForFunction ) )
	{
		Error::Error_1100_AlreadyDefined( funcStartIter, this, m_AliasForFunction );
		return false;
	}

	UnityNode * un = ng.CreateUnityNode( m_AliasForFunction, // name
										 dirNodes,
										 fullOutputPath,
										 outputPattern,
										 numFilesV ? numFilesV->GetInt() : 1,
										 precompiledHeader,
										 filesToExclude );
	ASSERT( un ); (void)un;

	return true;
}

//------------------------------------------------------------------------------
