// FunctionLibrary
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "FunctionLibrary.h"
#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/Flog.h>
#include <Tools/FBuild/FBuildCore/BFF/BFFIterator.h>
#include <Tools/FBuild/FBuildCore/BFF/BFFParser.h>
#include <Tools/FBuild/FBuildCore/BFF/BFFStackFrame.h>
#include <Tools/FBuild/FBuildCore/BFF/BFFVariable.h>
#include <Tools/FBuild/FBuildCore/Graph/NodeGraph.h>
#include <Tools/FBuild/FBuildCore/Graph/DirectoryListNode.h>
#include <Tools/FBuild/FBuildCore/Graph/LibraryNode.h>
#include <Tools/FBuild/FBuildCore/Graph/ObjectNode.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
FunctionLibrary::FunctionLibrary()
: Function( "Library" )
{
}

// AcceptsHeader
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionLibrary::AcceptsHeader() const
{
	return true;
}

// Commit
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionLibrary::Commit( const BFFIterator & funcStartIter ) const
{
	// make sure all required variables are defined
	const BFFVariable * outputLib;
	const BFFVariable * compiler;
	const BFFVariable * compilerOptions;
	const BFFVariable * compilerOutputPath;
	const BFFVariable * librarian;
	const BFFVariable * librarianOptions;
	if ( !GetString( funcStartIter, outputLib, ".LibrarianOutput", true ) ||
		 !GetString( funcStartIter, compiler, ".Compiler", true ) ||
		 !GetString( funcStartIter, compilerOptions, ".CompilerOptions", true ) ||
		 !GetString( funcStartIter, compilerOutputPath, ".CompilerOutputPath", true ) ||
		 !GetString( funcStartIter, librarian, ".Librarian", true ) ||
		 !GetString( funcStartIter, librarianOptions, ".LibrarianOptions", true ) )
	{
		return false;
	}

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();

	// find or create the compiler node
	Node * cn = ng.FindNode( compiler->GetValue() );
	FileNode * compilerNode = nullptr;
	if ( cn != nullptr )
	{
		if ( cn->IsAFile() == false )
		{
			Error::Error_1103_NotAFile( funcStartIter, this, "Compiler", cn->GetName(), cn->GetType() );
			return false;
		}
		compilerNode = cn->CastTo< FileNode >();
	}
	else
	{
		compilerNode = ng.CreateFileNode( compiler->GetValue() );
	}

	const bool isMSVC = compilerNode->GetName().EndsWithI( "\\cl.exe" );

	if ( isMSVC )
	{
		// must not specify use of precompiled header (must use the PCH specific options)
		if ( compilerOptions->GetValue().BeginsWith( "/Yc" ) || compilerOptions->GetValue().Find( " /Yc" ) )
		{
			Error::Error_1303_PCHCreateOptionOnlyAllowedOnPCH( funcStartIter, this, "/Yc", "CompilerOptions" );
			return false;
		}
	}

	// Precompiled Header support
	ObjectNode * precompiledHeaderNode = nullptr;
	const BFFVariable * pchInputFile = nullptr;
	const BFFVariable * pchOutputFile = nullptr;
	const BFFVariable * pchOptions = nullptr;
	if ( !GetString( funcStartIter, pchInputFile, ".PCHInputFile" ) ||
		 !GetString( funcStartIter, pchOutputFile, ".PCHOutputFile" ) ||
		 !GetString( funcStartIter, pchOptions, ".PCHOptions" ) )
	{
		return false;
	}
	if ( pchInputFile ) 
	{
		if ( !pchOutputFile || !pchOptions )
		{
			Error::Error_1300_MissingPCHArgs( funcStartIter, this );
			return false;
		}
		Node * pchInputNode = ng.FindNode( pchInputFile->GetValue() );
		if ( pchInputNode )
		{
			// is it a file?
			if ( pchInputNode->IsAFile() == false )
			{
				Error::Error_1103_NotAFile( funcStartIter, this, "PCHInputFile", pchInputNode->GetName(), pchInputNode->GetType() );
				return false;
			}
		}
		else
		{
			// Create input node
			pchInputNode = ng.CreateFileNode( pchInputFile->GetValue() );
		}

		if ( ng.FindNode( pchOutputFile->GetValue() ) )
		{
			Error::Error_1301_AlreadyDefinedPCH( funcStartIter, this, pchOutputFile->GetValue().Get() );
			return false;
		}

		uint32_t flags( 0 );
		if ( isMSVC )
		{
			flags |= ObjectNode::FLAG_MSVC;

			// sanity check arguments

			// PCH must have "Create PCH" (e.g. /Yc"PrecompiledHeader.h")
			if ( pchOptions->GetValue().Find( "/Yc" ) == nullptr )
			{
				Error::Error_1302_MissingPCHCompilerOption( funcStartIter, this, "/Yc", "PCHOptions" );
				return false;
			}
			// PCH must have "Precompiled Header to Use" (e.g. /Fp"PrecompiledHeader.pch")
			if ( pchOptions->GetValue().Find( "/Fp" ) == nullptr )
			{
				Error::Error_1302_MissingPCHCompilerOption( funcStartIter, this, "/Fp", "PCHOptions" );
				return false;
			}
			// PCH must have object output option (e.g. /Fo"PrecompiledHeader.obj")
			if ( pchOptions->GetValue().Find( "/Fo" ) == nullptr )
			{
				Error::Error_1302_MissingPCHCompilerOption( funcStartIter, this, "/Fo", "PCHOptions" );
				return false;
			}

			// Object using the PCH must have "Use PCH" option (e.g. /Yu"PrecompiledHeader.h")
			if ( compilerOptions->GetValue().Find( "/Yu" ) == nullptr )
			{
				Error::Error_1302_MissingPCHCompilerOption( funcStartIter, this, "/Yu", "CompilerOptions" );
				return false;
			}
			// Object using the PCH must have "Precompiled header to use" (e.g. /Fp"PrecompiledHeader.pch")
			if ( compilerOptions->GetValue().Find( "/Fp" ) == nullptr )
			{
				Error::Error_1302_MissingPCHCompilerOption( funcStartIter, this, "/Fp", "CompilerOptions" );
				return false;
			}
		}
		else if ( compilerNode->GetName().EndsWithI( "gcc" ) )
		{
			flags |= ObjectNode::FLAG_GCC;

			// TODO:B GCC arguments checks for PCH files
		}

		flags |= ObjectNode::FLAG_CREATING_PCH;

		precompiledHeaderNode = ng.CreateObjectNode( pchOutputFile->GetValue(),
													 pchInputNode,
													 compilerNode,
													 pchOptions->GetValue(),
													 nullptr,
													 flags );
	}

	Array< Node * > staticDeps( 32, true );

	// do we want to build a files in a directory?
	const BFFVariable * inputPath = BFFStackFrame::GetVar( ".CompilerInputPath" );
	if ( inputPath )
	{
		// get the optional pattern and recurse options related to InputPath
		const BFFVariable * patternVar = nullptr;
		if ( !GetString( funcStartIter, patternVar, ".CompilerInputPattern", false ) )
		{
			return false; // GetString will have emitted an error
		}
		AStackString<> defaultWildCard( "*.cpp" );
		const AString & pattern = patternVar ? patternVar->GetValue() : defaultWildCard;

		// recursive?  default to true
		bool recurse = true;
		if ( !GetBool( funcStartIter, recurse, ".CompilerInputPathRecurse", true, false ) )
		{
			return false; // GetBool will have emitted an error
		}

		// Support an exclusion path
		const BFFVariable * excludePathVar = nullptr;
		if ( !GetString( funcStartIter, excludePathVar, ".CompilerInputExcludePath", false ) )
		{
			return false; // GetString will have emitted an error
		}
		const AString & excludePath = excludePathVar ? excludePathVar->GetValue() : AString::GetEmpty();
		AStackString< 512 > fullExcludePath;
		if ( excludePath.IsEmpty() == false )
		{
			NodeGraph::CleanPath( excludePath, fullExcludePath );
			if ( fullExcludePath.EndsWith( '\\' ) == false )
			{
				fullExcludePath += '\\';
			}
		}

		Array< AString > paths;
		if ( inputPath->IsString() )
		{
			paths.Append( inputPath->GetValue() );
		}
		else if ( inputPath->IsArray() )
		{
			paths = inputPath->GetArray();
		}
		else
		{
			Error::Error_1050_PropertyMustBeOfType( funcStartIter, this, "CompilerInputPath", inputPath->GetType(), BFFVariable::VAR_STRING, BFFVariable::VAR_ARRAY );
			return false;
		}

		auto end = paths.End();
		for ( auto it = paths.Begin(); it != end; ++it )
		{
			const AString & path = *it;

			// clean paths
			AStackString< 512 > fullPath;
			NodeGraph::CleanPath( path, fullPath );
			if ( fullPath.EndsWith( '\\' ) == false )
			{
				fullPath += '\\';
			}

			// get node for the dir we depend on
			AStackString<> name;
			DirectoryListNode::FormatName( fullPath, pattern, recurse, fullExcludePath, name );
			Node * staticDep = ng.FindNode( name );
			if ( staticDep == nullptr )
			{
				staticDep = ng.CreateDirectoryListNode( name,
														fullPath,
														pattern,
														recurse,
														fullExcludePath );
			}
			else if ( staticDep->GetType() != Node::DIRECTORY_LIST_NODE )
			{
				Error::Error_1102_UnexpectedType( funcStartIter, this, "CompilerInputPath", staticDep->GetName(), staticDep->GetType(), Node::DIRECTORY_LIST_NODE );
				return false;
			}

			staticDeps.Append( staticDep );
		}
	}

	// do we want to build a specific list of files?
	if ( !GetNodeList( funcStartIter, ".CompilerInputFiles", staticDeps, false ) )
	{
		// helper will emit error
		return false;
	}

	// do we want to build files via a unity blob?
	const BFFVariable * inputUnity = nullptr;
	if ( !GetString( funcStartIter, inputUnity, ".CompilerInputUnity" ) )
	{
		return false;
	}
	if ( inputUnity )
	{
		Node * n = ng.FindNode( inputUnity->GetValue() );
		if ( n == nullptr )
		{
			Error::Error_1104_TargetNotDefined( funcStartIter, this, "CompilerInputUnity", inputUnity->GetValue() );
			return false;
		}
		if ( n->GetType() != Node::UNITY_NODE )
		{
			Error::Error_1102_UnexpectedType( funcStartIter, this, "CompilerInputUnity", inputUnity->GetValue(), n->GetType(), Node::UNITY_NODE );
			return false;
		}
		staticDeps.Append( n );
	}

	if ( staticDeps.IsEmpty() )
	{
		Error::Error_1006_NothingToBuild( funcStartIter, this );
		return false;
	}

	// Create library node which depends on the single file or list
	if ( ng.FindNode( outputLib->GetValue() ) )
	{
		Error::Error_1100_AlreadyDefined( funcStartIter, this, outputLib->GetValue() );
		return false;
	}
	Node * libNode = ng.CreateLibraryNode( outputLib->GetValue(),
						  staticDeps,
						  compilerNode,
						  compilerOptions->GetValue(),
						  compilerOutputPath->GetValue(),
						  librarian->GetValue(),
						  librarianOptions->GetValue(),
						  precompiledHeaderNode );

	return ProcessAlias( funcStartIter, libNode );
}

//------------------------------------------------------------------------------
