// FunctionForEach
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "FunctionForEach.h"

#include <Tools/FBuild/FBuildCore/BFF/BFFIterator.h>
#include <Tools/FBuild/FBuildCore/BFF/BFFParser.h>
#include <Tools/FBuild/FBuildCore/BFF/BFFStackFrame.h>
#include <Tools/FBuild/FBuildCore/BFF/BFFVariable.h>

#include <Core/Strings/AStackString.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
FunctionForEach::FunctionForEach()
: Function( "ForEach" )
{
}

// AcceptsHeader
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionForEach::AcceptsHeader() const
{
	return true;
}

// NeedsHeader
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionForEach::NeedsHeader() const
{
	return true;
}

//------------------------------------------------------------------------------
/*virtual*/ bool FunctionForEach::ParseFunction(
					const BFFIterator & functionNameStart,
					const BFFIterator & functionBodyStartToken, 
					const BFFIterator & functionBodyStopToken,
					const BFFIterator * functionHeaderStartToken,
					const BFFIterator * functionHeaderStopToken ) const
{
	// build array for each pair to loop through
	Array< AString >				localNames( 4, true );
	Array< const BFFVariable * >	arrayVars( 4, true );

	size_t loopLen = 0;

	// parse it all out
	BFFIterator pos( *functionHeaderStartToken );
	pos++; // skip opening token
	while ( pos < *functionHeaderStopToken )
	{
		pos.SkipWhiteSpace();
		if ( *pos != BFFParser::BFF_DECLARE_VAR_INTERNAL )
		{
			Error::Error_1200_ExpectedVar( pos, this );
			return false;
		}
		BFFIterator varNameStart( pos );
		pos++;
		if ( pos.IsAtValidVariableNameCharacter() == false )
		{
			Error::Error_1013_UnexpectedCharInVariableName( pos, this );
			return false;
		}
		pos.SkipVariableName();
		BFFIterator varNameEnd( pos );

		// sanity check it is a sensible length
		size_t varNameLen = varNameStart.GetDistTo( varNameEnd );
		if ( varNameLen > BFFParser::MAX_VARIABLE_NAME_LENGTH )
		{
			Error::Error_1014_VariableNameIsTooLong( varNameStart, (uint32_t)varNameLen, (uint32_t)BFFParser::MAX_VARIABLE_NAME_LENGTH );
			return false;
		}
		AStackString< BFFParser::MAX_VARIABLE_NAME_LENGTH > localName( varNameStart.GetCurrent(), varNameEnd.GetCurrent() );
		localNames.Append( localName );

		pos.SkipWhiteSpace();

		// check for "in" token
		bool foundIn = false;
		if ( *pos == 'i' )
		{
			pos++;
			if ( *pos == 'n' )
			{
				foundIn = true;
			}
		}
		if ( foundIn == false )
		{
			Error::Error_1201_MissingIn( pos, this );
			return false;
		}
		pos++;
		pos.SkipWhiteSpace();

		BFFIterator arrayVarNameBegin( pos );
		if ( *pos != BFFParser::BFF_DECLARE_VAR_INTERNAL )
		{
			Error::Error_1202_ExpectedVarFollowingIn( pos, this );
			return false;
		}
		pos++;
		if ( pos.IsAtValidVariableNameCharacter() == false )
		{
			Error::Error_1013_UnexpectedCharInVariableName( pos, this );
			return false;
		}
		pos.SkipVariableName();
		BFFIterator arrayVarNameEnd( pos );

		// sanity check it is a sensible length
		size_t arrayVarNameLen = arrayVarNameBegin.GetDistTo( arrayVarNameEnd );
		if ( arrayVarNameLen > BFFParser::MAX_VARIABLE_NAME_LENGTH )
		{
			Error::Error_1014_VariableNameIsTooLong( arrayVarNameBegin, (uint32_t)arrayVarNameLen, (uint32_t)BFFParser::MAX_VARIABLE_NAME_LENGTH );
			return false;
		}
		AStackString< BFFParser::MAX_VARIABLE_NAME_LENGTH > arrayVarName( arrayVarNameBegin.GetCurrent(), arrayVarNameEnd.GetCurrent() );

		const BFFVariable * var = BFFStackFrame::GetVar( arrayVarName );
		if ( var == nullptr )
		{
			Error::Error_1009_UnknownVariable( arrayVarNameBegin, this );
			return false;
		}
		if ( var->IsArray() == false )
		{
			Error::Error_1050_PropertyMustBeOfType( arrayVarNameBegin, this, arrayVarName.Get(), var->GetType(), BFFVariable::VAR_ARRAY );
			return false;
		}

		// is this the first variable?
		if ( loopLen == 0 )
		{
			// it will set the length of looping
			loopLen = var->GetArray().GetSize();
			if ( loopLen == 0 )
			{
				// doesn't make sense to loop through nothing
				Error::Error_1004_EmptyStringPropertyNotAllowed( arrayVarNameBegin, this, arrayVarName.Get() );
				return false;
			}
		}
		else
		{
			// variables after the first must match length of the first
			if ( loopLen != var->GetArray().GetSize() )
			{
				Error::Error_1204_LoopVariableLengthsDiffer( arrayVarNameBegin, this, arrayVarName.Get(), (uint32_t)var->GetArray().GetSize(), (uint32_t)loopLen );
				return false;
			}
		}
		arrayVars.Append( var );

		// skip optional separator
		pos.SkipWhiteSpace();
		if ( *pos == ',' )
		{
			pos++;
		}
	}

	ASSERT( localNames.GetSize() == arrayVars.GetSize() );
	ASSERT( loopLen > 0 );

	for ( uint32_t i=0; i<loopLen; ++i )
	{
		// create local loop variables
		BFFStackFrame loopStackFrame;
		for ( uint32_t j=0; j<localNames.GetSize(); ++j )
		{
			loopStackFrame.SetVar( localNames[ j ], arrayVars[ j ]->GetArray()[ i ] );
		}

		// parse the function body
		BFFParser subParser;
		BFFIterator subIter( functionBodyStartToken );
		subIter++; // skip opening token
		subIter.SetMax( functionBodyStopToken.GetCurrent() ); // limit to closing token
		if ( subParser.Parse( subIter ) == false )
		{
			return false;
		}

		// complete the function
		if ( Commit( functionNameStart ) == false )
		{
			return false;
		}
	}

	return true;
}

//------------------------------------------------------------------------------
