// FunctionExecutable
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "FunctionExecutable.h"
#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/Flog.h>
#include <Tools/FBuild/FBuildCore/BFF/BFFStackFrame.h>
#include <Tools/FBuild/FBuildCore/BFF/BFFVariable.h>
#include <Tools/FBuild/FBuildCore/Graph/AliasNode.h>
#include <Tools/FBuild/FBuildCore/Graph/LinkerNode.h>
#include <Tools/FBuild/FBuildCore/Graph/NodeGraph.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
FunctionExecutable::FunctionExecutable()
: Function( "Executable" )
{
}

// AcceptsHeader
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionExecutable::AcceptsHeader() const
{
	return true;
}

// Commit
//------------------------------------------------------------------------------
/*virtual*/ bool FunctionExecutable::Commit( const BFFIterator & funcStartIter ) const
{
	// make sure all required variables are defined
	const BFFVariable * linker = BFFStackFrame::GetVar( ".Linker" );
	if ( linker == nullptr )
	{
		Error::Error_1101_MissingProperty( funcStartIter, this, AStackString<>( "Linker" ) );
		return false;
	}
	const BFFVariable * linkerOutput = BFFStackFrame::GetVar( ".LinkerOutput" );
	if ( linkerOutput == nullptr )
	{
		Error::Error_1101_MissingProperty( funcStartIter, this, AStackString<>( "LinkerOutput" ) );
		return false;
	}
	const BFFVariable * linkerOptions = BFFStackFrame::GetVar( ".LinkerOptions" );
	if ( linkerOptions == nullptr )
	{
		Error::Error_1101_MissingProperty( funcStartIter, this, AStackString<>( "LinkerOptions" ) );
		return false;
	}
	const BFFVariable * libraries = BFFStackFrame::GetVar( ".Libraries" );
	if ( libraries == nullptr )
	{
		Error::Error_1101_MissingProperty( funcStartIter, this, AStackString<>( "Libraries" ) );
		return false;
	}

	NodeGraph & ng = FBuild::Get().GetDependencyGraph();

	// we'll build a list of libraries to link
	Array< Node * > libraryNodes( 64, true );

	// has a list been provided?
	if ( libraries->IsArray() )
	{
		// convert string list to nodes
		const Array< AString > & libraryNames = libraries->GetArray();
		for ( Array< AString >::ConstIter it = libraryNames.Begin();
			  it != libraryNames.End();
			  it++ )
		{
			if ( DependOnNode( funcStartIter, *it, libraryNodes ) == false )
			{
				return false; // DependOnNode will have emitted an error
			}
		}
	}
	else if ( libraries->IsString() )
	{
		// handle this one node
		if ( DependOnNode( funcStartIter, libraries->GetValue(), libraryNodes ) == false )
		{
				return false; // DependOnNode will have emitted an error
		}
	}
	else
	{
		Error::Error_1050_PropertyMustBeOfType( funcStartIter, this, "Libraries", libraries->GetType(), BFFVariable::VAR_STRING, BFFVariable::VAR_ARRAY );
		return false;
	}

	// Check for existing node
	if ( ng.FindNode( linkerOutput->GetValue() ) )
	{
		Error::Error_1100_AlreadyDefined( funcStartIter, this, linkerOutput->GetValue() );
		return false;
	}

	// make node for exe
	Node * n = ng.CreateLinkerNode( linkerOutput->GetValue(),
							libraryNodes,
							linker->GetValue(),
							linkerOptions->GetValue() );

	return ProcessAlias( funcStartIter, n );
}

// DependOnNode
//------------------------------------------------------------------------------
bool FunctionExecutable::DependOnNode( const BFFIterator & iter, const AString & nodeName, Array< Node * > & nodes ) const
{
	NodeGraph & ng = FBuild::Get().GetDependencyGraph();
	Node * node = ng.FindNode( nodeName );

	// does it exist?
	if ( node != nullptr )
	{
		// process it
		return DependOnNode( iter, node, nodes );
	}

	// node not found - create a new FileNode, assuming we are
	// linking against an externally built library
	node = ng.CreateFileNode( nodeName );
	nodes.Append( node );
	return true;
}

// DependOnNode
//------------------------------------------------------------------------------
bool FunctionExecutable::DependOnNode( const BFFIterator & iter, Node * node, Array< Node * > & nodes ) const
{
	ASSERT( node );

	// a previously declared library?
	if ( node->GetType() == Node::LIBRARY_NODE )
	{
		// can link directly to it
		nodes.Append( node );
		return true;
	}

	// a previously declared external file?
	if ( node->GetType() == Node::FILE_NODE )
	{
		// can link directy against it
		nodes.Append( node );
		return true;
	}

	// a group (alias)?
	if ( node->GetType() == Node::ALIAS_NODE )
	{
		// handle all targets in alias
		AliasNode * an = node->CastTo< AliasNode >();
		const Array< Node * > & aliasNodeList = an->GetStaticDependencies();
		const Array< Node * >::Iter end = aliasNodeList.End();
		for ( Array< Node * >::Iter it = aliasNodeList.Begin();
			  it != end; 
			  ++it )
		{
			if ( DependOnNode( iter, *it, nodes ) == false )
			{
				return false; // something went wrong lower down
			}
		}
		return true; // all nodes in group handled ok
	}

	// don't know how to handle this type of node
	Error::Error_1005_UnsupportedNodeType( iter, this, "Libraries", node->GetName(), node->GetType() );
	return false;
}

//------------------------------------------------------------------------------
