// BFFStackFrame
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "BFFStackFrame.h"
#include "BFFVariable.h"
#include <Core/Strings/AStackString.h>

//
/*static*/ BFFStackFrame * BFFStackFrame::s_StackHead = nullptr;

// CONSTRUCTOR
//------------------------------------------------------------------------------
BFFStackFrame::BFFStackFrame()
: m_Variables( 32, true )
{
	// hook into top of stack chain
	m_Next = s_StackHead;
	s_StackHead = this;
}

// DESTRUCTOR
//------------------------------------------------------------------------------
BFFStackFrame::~BFFStackFrame()
{
	// unhook from top of stack chain
	ASSERT( s_StackHead == this );
	s_StackHead = m_Next;

	// free all variables we own
	Array< BFFVariable * >::Iter i = m_Variables.Begin();
	Array< BFFVariable * >::Iter end = m_Variables.End();
	for( ; i < end ; ++i )
	{
		delete *i;
	}
}

// SetVar
//------------------------------------------------------------------------------
/*static*/ void BFFStackFrame::SetVar( const AString & name,
									   const AString & value )
{
	BFFVariable * var = s_StackHead->GetVarMutableNoRecurse( name );
	if ( var )
	{
		var->SetValue( value );
		return;
	}

	// variable not found at this level, so create it
	s_StackHead->m_Variables.Append( new BFFVariable( name, value ) );
}

// SetVar
//------------------------------------------------------------------------------
/*static*/ void BFFStackFrame::SetVarArray( const AString & name,
											const Array< AString > & values )
{
	BFFVariable * var = s_StackHead->GetVarMutableNoRecurse( name );
	if ( var )
	{
		var->SetValue( values );
		return;
	}

	// variable not found at this level, so create it
	s_StackHead->m_Variables.Append( new BFFVariable( name, values ) );
}

// SetVarBool
//------------------------------------------------------------------------------
/*static*/ void BFFStackFrame::SetVarBool( const AString & name, bool value )
{
	BFFVariable * var = s_StackHead->GetVarMutableNoRecurse( name );
	if ( var )
	{
		var->SetValue( value );
		return;
	}

	// variable not found at this level, so create it
	s_StackHead->m_Variables.Append( new BFFVariable( name, value ) );
}

// SetVarBool
//------------------------------------------------------------------------------
/*static*/ void BFFStackFrame::SetVarInt( const AString & name, int value )
{
	BFFVariable * var = s_StackHead->GetVarMutableNoRecurse( name );
	if ( var )
	{
		var->SetValue( value );
		return;
	}

	// variable not found at this level, so create it
	s_StackHead->m_Variables.Append( new BFFVariable( name, value ) );
}

// GetVar
//------------------------------------------------------------------------------
/*static*/ const BFFVariable * BFFStackFrame::GetVar( const char * name )
{
	AStackString<> strName( name );
	return GetVar( strName );
}

// GetVar
//------------------------------------------------------------------------------
/*static*/ const BFFVariable * BFFStackFrame::GetVar( const AString & name )
{
	// we shouldn't be calling this if there aren't any stack frames
	ASSERT( s_StackHead );

	// recurse up the stack
	return s_StackHead->GetVariableRecurse( name );
}

// GetVariableRecurse
//------------------------------------------------------------------------------
const BFFVariable * BFFStackFrame::GetVariableRecurse( const AString & name ) const
{
	// look at this scope level
	Array< BFFVariable * >::Iter i = m_Variables.Begin();
	Array< BFFVariable * >::Iter end = m_Variables.End();
	for( ; i < end ; ++i )
	{
		if ( ( *i )->GetName() == name )
		{
			return *i;
		}
	}
	
	// look at parent
	if ( m_Next )
	{
		return m_Next->GetVariableRecurse( name );
	}

	// not found
	return nullptr;
}

// GetVarAny
//------------------------------------------------------------------------------
/*static*/ const BFFVariable * BFFStackFrame::GetVarAny( const AString & name )
{
	// we shouldn't be calling this if there aren't any stack frames
	ASSERT( s_StackHead );

	// recurse up the stack
	return s_StackHead->GetVariableRecurse( name, BFFVariable::VAR_ANY );
}

// GetVariableRecurse
//------------------------------------------------------------------------------
const BFFVariable * BFFStackFrame::GetVariableRecurse( const AString & nameOnly, 
												 BFFVariable::VarType type ) const
{
	// look at this scope level
	Array< BFFVariable * >::Iter i = m_Variables.Begin();
	Array< BFFVariable * >::Iter end = m_Variables.End();
	for( ; i < end ; ++i )
	{
		// if name only (minus type prefix ) length matches
		if ( ( *i )->GetName().GetLength() == ( nameOnly.GetLength() + 1 ) )
		{
			//types match?
			if ( ( type == BFFVariable::VarType::VAR_ANY ) ||
				 ( type == ( *i )->GetType() ) )
			{
				// compare names
				if ( nameOnly == ( ( *i )->GetName().Get() + 1 ) )
				{
					return *i;
				}
			}
		}
	}
	
	// look at parent
	if ( m_Next )
	{
		return m_Next->GetVariableRecurse( nameOnly, type );
	}

	// not found
	return nullptr;
}

// GetVarMutableNoRecurse
//------------------------------------------------------------------------------
BFFVariable * BFFStackFrame::GetVarMutableNoRecurse( const AString & name )
{
	ASSERT( s_StackHead ); // we shouldn't be calling this if there aren't any stack frames

	// look at this scope level
	Array< BFFVariable * >::Iter i = m_Variables.Begin();
	Array< BFFVariable * >::Iter end = m_Variables.End();
	for( ; i < end ; ++i )
	{
		if ( ( *i )->GetName() == name )
		{
			return *i;
		}
	}

	return nullptr;
}

//------------------------------------------------------------------------------
