// BFFParser
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "BFFParser.h"
#include "BFFIterator.h"
#include "BFFStackFrame.h"

#include "../FLog.h"
#include "Functions/Function.h"

#include <Core/Env/Assert.h>
#include <Core/FileIO/FileStream.h>
#include <Core/Strings/AStackString.h>
#include <Core/Time/Timer.h>
#include <Core/Tracing/Tracing.h>

#include <stdio.h>

// CONSTRUCTOR
//------------------------------------------------------------------------------
BFFParser::BFFParser()
: m_SeenAVariable( false )
{
}

// DESTRUCTOR
//------------------------------------------------------------------------------
BFFParser::~BFFParser()
{
}

// Load
//------------------------------------------------------------------------------
bool BFFParser::Parse( const char * dataWithSentinel, 
					   uint32_t sizeExcludingSentinel, 
					   const char * fileName )
{
	// data should be 1 bytes larger than size, with a sentinel
	ASSERT( dataWithSentinel[ sizeExcludingSentinel ] == '\000' );

	//m_FileName = fileName;

	// parse it
	BFFStackFrame stackFrame;
	BFFIterator iter( dataWithSentinel, sizeExcludingSentinel, fileName );
	if ( Parse( iter ) == false )
	{
		// Parse will have emitted an error
		return false;
	}

	return true;
}

// Parse
//------------------------------------------------------------------------------
bool BFFParser::Parse( BFFIterator & iter )
{
	for (;;)
	{
		iter.SkipWhiteSpace();

		// is this a comment?
		if ( iter.IsAtComment() )
		{
			iter.SkipComment();
			continue;
		}

		const char c = *iter;
		switch ( c )
		{
			case BFF_DECLARE_VAR_INTERNAL:
			{
				if ( ParseNamedVariableDeclaration( iter ) == false )
				{
					return false;
				}
				continue;
			}
			case BFF_VARIABLE_CONCATENATION:
			{
				// concatenation to last used variable
				if ( ParseUnnamedVariableConcatenation( iter ) == false )
				{
					return false;
				}
				continue;
			}
			default:
			{
				if ( iter.IsAtValidFunctionNameCharacter() )
				{
					if ( ParseFunction( iter ) == false )
					{
						return false;
					}
					continue;
				}
			}
		}

		iter.SkipWhiteSpace();
		if ( iter.IsAtEnd() == false )
		{
			Error::Error_1010_UnknownConstruct( iter );
			return false;
		}

		break;  // cleanly hit end of file
	}

	return true;
}

// ParseUnnamedVariableConcatenation
//------------------------------------------------------------------------------
bool BFFParser::ParseUnnamedVariableConcatenation( BFFIterator & iter )
{
	ASSERT( *iter == BFF_VARIABLE_CONCATENATION );

	// have we assigned a variable before?
	if ( m_SeenAVariable == false )
	{
		Error::Error_1011_UnnamedConcatMustFollowAssignment( iter );
		return false;
	}

	return ParseVariableDeclaration( iter, m_LastVarNameStart, m_LastVarNameEnd );
}

// ParseNamedVariableDeclaration
//------------------------------------------------------------------------------
bool BFFParser::ParseNamedVariableDeclaration( BFFIterator & iter )
{
	// skip over the declaration symbol
	ASSERT( *iter == BFF_DECLARE_VAR_INTERNAL );
	m_LastVarNameStart = iter; // include type token in var name
	iter++;

	// make sure we haven't hit the end of the file
	if ( iter.IsAtEnd() )
	{
		Error::Error_1012_UnexpectedEndOfFile( iter );
		return false;
	}

	// make sure immediately after the symbol starts a variable name
	if ( iter.IsAtValidVariableNameCharacter() == false )
	{
		Error::Error_1013_UnexpectedCharInVariableName( iter, nullptr );
		return false;
	}

	// find the end of the variable name
	iter.SkipVariableName();
	if ( iter.IsAtEnd() )
	{
		Error::Error_1012_UnexpectedEndOfFile( iter );
		return false;
	}
	m_LastVarNameEnd = iter;

	// sanity check it is a sensible length
	size_t varNameLen = m_LastVarNameStart.GetDistTo( m_LastVarNameEnd );
	if ( varNameLen > MAX_VARIABLE_NAME_LENGTH )
	{
		Error::Error_1014_VariableNameIsTooLong( iter, (uint32_t)varNameLen, (uint32_t)MAX_VARIABLE_NAME_LENGTH );
		return false;
	}

	// find the start of the assignment
	iter.SkipWhiteSpaceAndComments();
	if ( iter.IsAtEnd() )
	{
		Error::Error_1012_UnexpectedEndOfFile( iter );
		return false;
	}

	return ParseVariableDeclaration( iter, m_LastVarNameStart, m_LastVarNameEnd );
}

// ParseVariableDeclaration
//------------------------------------------------------------------------------
bool BFFParser::ParseVariableDeclaration( BFFIterator & iter, const BFFIterator & varNameStart,
															  const BFFIterator & varNameEnd )
{
	// look for an appropriate operator
	BFFIterator operatorIter( iter );
	bool concatenation = false;
	if ( *iter == BFF_VARIABLE_ASSIGNMENT )
	{
		// assignment
	}
	else if ( *iter == BFF_VARIABLE_CONCATENATION )
	{
		// concatenation
		concatenation = true;
	}
	else
	{
		Error::Error_1016_UnexepectedCharFollowingVariableName( iter );
		return false;
	}

	// skip the assignment symbol and whitespace
	iter++;
	iter.SkipWhiteSpaceAndComments();
	if ( iter.IsAtEnd() )
	{
		Error::Error_1012_UnexpectedEndOfFile( iter );
		return false;
	}

	AStackString< 64 > varName( varNameStart.GetCurrent(), varNameEnd.GetCurrent() );

	char openToken = *iter;
	char closeToken = 0;
	bool ok = false;
	if ( ( openToken == '"' ) || ( openToken == '\'' ) )
	{
		closeToken = openToken;
		ok = true;
	}
	else if ( openToken == BFF_FUNCTION_BODY_OPEN )
	{
		closeToken = BFF_FUNCTION_BODY_CLOSE;
		ok = true;
	}
	else if ( ( openToken >= '0' ) && ( openToken <= '9' ) )
	{
		if ( concatenation )
		{
			Error::Error_1027_CannotConcatenate( operatorIter, varName, BFFVariable::VAR_ANY, BFFVariable::VAR_INT );
			return false;
		}

		// integer value?
		BFFIterator startIntValue( iter );
		while ( iter.IsAtEnd() == false )
		{
			iter++;
			if ( ( *iter < '0' ) || ( *iter > '9' ) )
			{
				break; // end of integer
			}
		}
		if ( startIntValue.GetDistTo( iter ) > 10 )
		{
			Error::Error_1018_IntegerValueCouldNotBeParsed( startIntValue );
			return false;
		}
		AStackString<> intAsString( startIntValue.GetCurrent(), iter.GetCurrent() );
		int i = 0;
		if ( sscanf_s( intAsString.Get(), "%i", &i ) != 1 )
		{
			Error::Error_1018_IntegerValueCouldNotBeParsed( startIntValue );
			return false;
		}
		return StoreVariableInt( varNameStart.GetCurrent(), varNameEnd.GetCurrent(), i );
	}
	else if ( ( *iter == 't' ) || ( *iter == 'f' ) )
	{
		// might be 'true' or 'false'
		BFFIterator startBoolValue( iter );
		if ( iter.ParseToNext( 'e' ) == true )
		{
			iter++;
			if ( ( startBoolValue.GetDistTo( iter ) <= 5 ) )
			{
				AStackString<8> value( startBoolValue.GetCurrent(), iter.GetCurrent() );
				if ( value == "true" )
				{
					if ( concatenation )
					{
						Error::Error_1027_CannotConcatenate( operatorIter, varName, BFFVariable::VAR_ANY, BFFVariable::VAR_BOOL );
						return false;
					}
					return StoreVariableBool( varNameStart.GetCurrent(), varNameEnd.GetCurrent(), true );
				}
				else if ( value == "false" )
				{
					if ( concatenation )
					{
						Error::Error_1027_CannotConcatenate( operatorIter, varName, BFFVariable::VAR_ANY, BFFVariable::VAR_BOOL );
						return false;
					}
					return StoreVariableBool( m_LastVarNameStart.GetCurrent(), m_LastVarNameEnd.GetCurrent(), false );
				}
			}
		}

		// not a valid bool value
	}
	
	if ( !ok )
	{
		Error::Error_1017_UnexepectedCharInVariableValue( iter );
		return false;
	}

	bool result = true;

	// find closing token
	BFFIterator openTokenPos( iter );
	openTokenPos++; // more start iter to first char of value
	if ( openToken == BFF_FUNCTION_BODY_OPEN )
	{
		if ( iter.ParseToMatchingBrace( openToken, closeToken ) )
		{
			result = StoreVariableArray( varNameStart.GetCurrent(), varNameEnd.GetCurrent(), 
										 openTokenPos, iter, operatorIter );
		}
	}
	else
	{
		ASSERT( ( openToken == '\'' ) || ( openToken == '"' ) );
		iter.SkipString( closeToken );
		if ( *iter == closeToken )
		{
			result = StoreVariableString( varNameStart.GetCurrent(), varNameEnd.GetCurrent(), 
										  openTokenPos, iter, operatorIter );
		}
	}

	if ( result )
	{
		m_SeenAVariable = true;
		iter++; // skip over the end token
	}

	// StoreVariable will have emitted an error if there was one
	return result;
}

// ParseFunction
//------------------------------------------------------------------------------
bool BFFParser::ParseFunction( BFFIterator & iter )
{
	ASSERT( iter.IsAtValidFunctionNameCharacter() );

	// for variables to be used by this function
	BFFStackFrame stackFrame;

	BFFIterator functionNameStart( iter );
	iter.SkipFunctionName();
	if ( iter.IsAtEnd() )
	{
		Error::Error_1012_UnexpectedEndOfFile( iter );
		return false;
	}

	// check length
	if ( functionNameStart.GetDistTo( iter ) > MAX_FUNCTION_NAME_LENGTH )
	{
		// if it's too long, then it can't be a valid function
		Error::Error_1015_UnknownFunction( functionNameStart );
		return false;
	}

	// store function name
	AStackString<MAX_FUNCTION_NAME_LENGTH> functionName( functionNameStart.GetCurrent(), iter.GetCurrent() );
	const Function * func = Function::Find( functionName );
	if ( func == nullptr )
	{
		Error::Error_1015_UnknownFunction( functionNameStart );
		return false;
	}
	iter.SkipWhiteSpace();

	if ( func->IsUnique() && func->GetSeen() )
	{
		Error::Error_1020_FunctionCanOnlyBeInvokedOnce( functionNameStart, func );
		return false;
	}
	func->SetSeen();

	FLOG_INFO( "Function call '%s'", functionName.Get() );

	// header, or body?
	bool hasHeader = false;
	BFFIterator functionArgsStartToken( iter );
	BFFIterator functionArgsStopToken( iter );
	if ( *iter == BFF_FUNCTION_ARGS_OPEN )
	{
		// can this function accept a header?
		if ( func->AcceptsHeader() == false )
		{
			Error::Error_1021_UnexpectedHeaderForFunction( iter, func );
			return false;
		}

		// args
		if ( iter.ParseToMatchingBrace( BFF_FUNCTION_ARGS_OPEN, BFF_FUNCTION_ARGS_CLOSE ) == false )
		{
			Error::Error_1022_MissingFunctionHeaderCloseToken( functionArgsStartToken, func );
			return false;
		}
		functionArgsStopToken = iter;
		hasHeader = true;
		iter++; // skip over closing token	
		iter.SkipWhiteSpaceAndComments();
	}

	if ( func->NeedsHeader() && ( hasHeader == false ) )
	{
		Error::Error_1023_FunctionRequiresAHeader( iter, func );
		return false;
	}

	// find body
	if ( *iter != BFF_FUNCTION_BODY_OPEN )
	{
		Error::Error_1024_FunctionRequiresABody( functionNameStart, func );
		return false;
	}

	BFFIterator functionBodyStartToken( iter );
	if ( iter.ParseToMatchingBrace( BFF_FUNCTION_BODY_OPEN, BFF_FUNCTION_BODY_CLOSE ) == false )
	{
		Error::Error_1025_MissingFunctionBodyCloseToken( functionBodyStartToken, func );
		return false;
	}

	BFFIterator functionBodyStopToken( iter );
	iter++;

	if ( hasHeader )
	{
		return func->ParseFunction( functionNameStart,
									functionBodyStartToken, functionBodyStopToken,
									&functionArgsStartToken, &functionArgsStopToken );
	}
	else
	{
		return func->ParseFunction( functionNameStart,
									functionBodyStartToken, functionBodyStopToken,
									nullptr, nullptr );
	}
}

// StoreVariableString
//------------------------------------------------------------------------------
bool BFFParser::StoreVariableString( const char * varNameStart, const char * varNameEnd, 
									 const BFFIterator & valueStart, const BFFIterator & valueEnd,
									 const BFFIterator & operatorIter )
{
	ASSERT( varNameStart );
	ASSERT( varNameEnd );
	AStackString< MAX_VARIABLE_NAME_LENGTH > name( varNameStart, varNameEnd );

	// are we concatenating?
	const BFFVariable * varToConcat = nullptr;
	if ( *operatorIter == BFF_VARIABLE_CONCATENATION )
	{
		// find existing
		varToConcat = BFFStackFrame::GetVar( name );
		if ( varToConcat == nullptr )
		{
			Error::Error_1026_VariableNotFoundForConcatenation( operatorIter, name );
			return false;
		}

		// make sure existing is a string
		if ( varToConcat->IsString() == false )
		{
			Error::Error_1027_CannotConcatenate( operatorIter, name, BFFVariable::VAR_STRING, varToConcat->GetType() );
			return false;
		}
	}

	// Register this variable
	AStackString< 2048 > value;

	// unescape and subsitute embedded variables
	if ( PerformVariableSubstitutions( valueStart, valueEnd, value ) == false )
	{
		return false;
	}

	AStackString< MAX_VARIABLE_NAME_LENGTH > finalValue;
	if ( varToConcat )
	{
		finalValue = varToConcat->GetValue();
	}
	finalValue += value.Get(); 

	BFFStackFrame::SetVar( name, finalValue );
	FLOG_INFO( "Registered <string> variable '%s' with value '%s'", name.Get(), finalValue.Get() );

	return true;
}

// StoreVariableString
//------------------------------------------------------------------------------
bool BFFParser::StoreVariableArray( const char * varNameStart, const char * varNameEnd, 
									const BFFIterator & valueStart, const BFFIterator & valueEnd,
									const BFFIterator & operatorIter )
{
	ASSERT( varNameStart );
	ASSERT( varNameEnd );
	AStackString< MAX_VARIABLE_NAME_LENGTH > name( varNameStart, varNameEnd );

	Array< AString > values( 32, true );

	// are we concatenating?
	if ( *operatorIter == BFF_VARIABLE_CONCATENATION )
	{
		// find existing
		const BFFVariable * var = BFFStackFrame::GetVar( name );
		if ( var == nullptr )
		{
			Error::Error_1026_VariableNotFoundForConcatenation( operatorIter, name );
			return false;
		}

		// make sure existing is an array
		if ( var->IsArray() == false )
		{
			Error::Error_1027_CannotConcatenate( operatorIter, name, BFFVariable::VAR_ARRAY, var->GetType() );
			return false;
		}

		// get values to start with
		values = var->GetArray();
	}

	// Parse array of variables
	BFFIterator iter( valueStart );
	while ( iter < valueEnd )
	{
		iter.SkipWhiteSpaceAndComments();
		const char c = *iter;
		if ( ( c != '"' ) && ( c != '\'' ) )
		{
			Error::Error_1001_MissingStringStartToken( iter, nullptr );
			return false;
		}
		BFFIterator elementValueStart( iter );
		iter.SkipString( c );
		ASSERT( iter.GetCurrent() <= valueEnd.GetCurrent() ); // should not be in this function if string is not terminated
		elementValueStart++; // move to start of actual content
		AStackString< 2048 > elementValue;

		// unescape and subsitute embedded variables
		if ( PerformVariableSubstitutions( elementValueStart, iter, elementValue ) == false )
		{
			return false;
		}

		values.Append( elementValue );

		iter++; // pass closing quote
		iter.SkipWhiteSpaceAndComments();
		if ( *iter == ',' ) // comma seperators are optional
		{ 
			iter++; 
		}

		// continue looking for more vars...
	}

	// Register this variable
	BFFStackFrame::SetVarArray( name, values );
	FLOG_INFO( "Registered <array> variable '%s' with %u elements", name.Get(), values.GetSize() );

	return true;
}

// StoreVariableBool
//------------------------------------------------------------------------------
bool BFFParser::StoreVariableBool( const char * varNameStart, const char * varNameEnd, bool value )
{
	// Register this variable
	AStackString< MAX_VARIABLE_NAME_LENGTH > name( varNameStart, varNameEnd );
	BFFStackFrame::SetVarBool( name, value );

	FLOG_INFO( "Registered <bool> variable '%s' with value '%s'", name.Get(), value ? "true" : "false" );

	return true;
}

// StoreVariableInt
//------------------------------------------------------------------------------
bool BFFParser::StoreVariableInt( const char * varNameStart, const char * varNameEnd, int value )
{
	AStackString< MAX_VARIABLE_NAME_LENGTH > name( varNameStart, varNameEnd );
	BFFStackFrame::SetVarInt( name, value );

	FLOG_INFO( "Registered <int> variable '%s' with value '%i'", name.Get(), value );

	return true;
}

// PerformVariableSubstitutions
//------------------------------------------------------------------------------
/*static*/ bool BFFParser::PerformVariableSubstitutions( const BFFIterator & startIter,
											  const BFFIterator & endIter,
											  AString & value )
{
	AStackString< 4096 > output;

	BFFIterator src( startIter );
	BFFIterator end( endIter );

	while ( src < end )
	{
		switch ( *src )
		{
			case '^':
			{
				src++; // skip escape char
				if ( src < end )
				{
					output += '"'; // append escaped character
				}
				break;
			}
			case '$':
			{
				BFFIterator firstToken( src );
				src++; // skip opening $

				// find matching $
				BFFIterator startName( src );
				const char * endName = nullptr;
				while ( src < end )
				{
					if ( *src == '$' )
					{
						endName = src.GetCurrent();
						break;
					}
					src++;
				}
				if ( ( endName == nullptr ) ||
					 ( ( endName - startName.GetCurrent() ) < 1 ) )
				{
					Error::Error_1028_MissingVariableSubstitutionEnd( firstToken );
					return false; 
				}
				AStackString< MAX_VARIABLE_NAME_LENGTH > varName( startName.GetCurrent(), endName );
				const BFFVariable * var = BFFStackFrame::GetVarAny( varName );
				if ( var == nullptr )
				{
					Error::Error_1009_UnknownVariable( startName, nullptr );
					return false; 
				}
				if ( var->IsString() == false )
				{
					Error::Error_1029_VariableForSubstitutionIsNotAString( startName, varName, var->GetType() );
					return false; 
				}
				output += var->GetValue();
				break;
			}
			default:
			{
				output += *src;
				break;
			}
		}
		src++;
	}

	value = output;
	return true;
}

//------------------------------------------------------------------------------
