// FBuild.cpp : Defines the entry point for the console application.
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include <Tools/FBuild/FBuildCore/FBuild.h>
#include <Tools/FBuild/FBuildCore/FLog.h>

#include <Core/Containers/Array.h>
#include <Core/Strings/AStackString.h>
#include <Core/Tracing/Tracing.h>

#include <windows.h> 
#include <stdio.h>

// Defines
//------------------------------------------------------------------------------
#define VERSION_STRING "v0.53"

// Return Codes
//------------------------------------------------------------------------------
enum ReturnCodes
{
	FBUILD_OK								= 0,
	FBUILD_BUILD_UNKNOWN_TARGET				= -1,
	FBUILD_ERROR_LOADING_BFF				= -2,
	FBUILD_BAD_ARGS							= -3
};

// Headers
//------------------------------------------------------------------------------
void DisplayHelp();
void DisplayVersion();
BOOL CtrlHandler( DWORD fdwCtrlType ); // Handle Ctrl+C etc

// main
//------------------------------------------------------------------------------
int main(int argc, char * argv[])
{
	VERIFY( SetConsoleCtrlHandler( (PHANDLER_ROUTINE)CtrlHandler, TRUE ) ); // Register

	// handle cmd line args
	Array< AString > targets( 8, true );
	bool cleanBuild = false;
	bool verbose = false;
	bool progressBar = true;
	bool useCacheRead = false;
	bool useCacheWrite = false;
	bool allowDistributed = false;
	bool showSummary = false;
	bool noOutputBuffering = false;
	int32_t numWorkers = -1;
	const char * configFile = nullptr;
	for ( int32_t i=1; i<argc; ++i ) // start from 1 to skip exe name
	{
		AStackString<> thisArg( argv[ i ] );

		// options start with a '-'
		if ( thisArg.BeginsWith( '-' ) )
		{
			if ( thisArg == "-cache" )
			{
				useCacheRead = true;
				useCacheWrite = true;
				continue;
			}
			else if ( thisArg == "-cacheread" )
			{
				useCacheRead = true;
				continue;
			}
			else if ( thisArg == "-cachewrite" )
			{
				useCacheWrite = true;
				continue;
			}
			else if ( thisArg == "-clean" )
			{
				cleanBuild = true;
				continue;
			}
			else if ( thisArg == "-config" )
			{
				int pathIndex = ( i + 1 );
				if ( pathIndex >= argc )
				{
					OUTPUT( "Error: Missing <path> for '-config' argument\n" );
					OUTPUT( "Try \"FBuild.exe -help\"\n" );
					return FBUILD_BAD_ARGS;
				}
				configFile = argv[ pathIndex ];
				i++; // skip extra arg we've consumed
				continue;
			}
			#ifdef DEBUG
				else if ( thisArg == "-debug" )
				{
					ASSERT( false && "Break due to '-debug' argument - attach debugger!" );
					continue;
				}
			#endif
			else if ( thisArg == "-dist" )
			{
				allowDistributed = true;
				continue;
			}
			else if ( thisArg == "-help" )
			{
				DisplayHelp();
				return FBUILD_OK; // exit app
			}
			else if ( thisArg.BeginsWith( "-j" ) &&
					  sscanf_s( thisArg.Get(), "-j%i", &numWorkers ) == 1 )
			{
				continue; // 'numWorkers' will contain value now
			}
			else if ( thisArg == "-nooutputbuffering" )
			{
				noOutputBuffering = true;
				continue;
			}
			else if ( thisArg == "-noprogress" )
			{
				progressBar = false;
				continue;
			}
			else if ( thisArg == "-summary" )
			{
				showSummary = true;
				continue;
			}
			else if ( thisArg == "-verbose" ) 
			{
				verbose = true;
				continue;
			}
			else if ( thisArg == "-version" ) 
			{
				DisplayVersion();
				return FBUILD_OK; // exit app
			}
			else if ( thisArg == "-vs" )
			{
				progressBar = false;
				noOutputBuffering = true;
				continue;
			}

			// can't use FLOG_ERROR as FLog is not initialized
			OUTPUT( "Error: Unknown argument '%s'\n", thisArg );
			OUTPUT( "Try \"FBuild.exe -help\"\n" );
			return FBUILD_BAD_ARGS;
		}
		else
		{
			// assume target
			targets.Append( AStackString<>( thisArg ) );
		}
	}

	if ( noOutputBuffering )
	{
		// prevent buffering, so output in Visual Studio is quick
		setvbuf(stdout, NULL, _IONBF, 0);
		setvbuf(stderr, NULL, _IONBF, 0);
	}

	FBuildOptions options;
	options.m_ShowProgress = progressBar;
	options.m_ShowInfo = verbose;
	options.m_UseCacheRead = useCacheRead;
	options.m_UseCacheWrite = useCacheWrite;
	if ( numWorkers >= 0 )
	{
		options.m_NumWorkerThreads = numWorkers;
	}
	options.m_ForceCleanBuild = cleanBuild; 
	options.m_AllowDistributed = allowDistributed;
	options.m_ShowSummary = showSummary;
	if ( configFile )
	{
		options.m_ConfigFile = configFile;
	}
	FBuild fBuild( options );

	if ( targets.IsEmpty() )
	{
		FLOG_INFO( "No target specified, defaulting to target 'all'" );
		targets.Append( AString( "all" ) );
	}

	// load the dependency graph if available
	if ( !fBuild.Initialize( FBuild::GetDependencyGraphFileName() ) )
	{
		return FBUILD_ERROR_LOADING_BFF;
	}

	bool result = true;

	// do the building
	if ( targets.GetSize() == 1 )
	{
		result = fBuild.Build( targets[ 0 ] );
	}
	else
	{
		FLOG_INFO( "Building %i targets", targets.GetSize() );

		// build each target specified
		// TODO:C Create a temporary node and build that instead of building each one
		//   (This will allow parallel building of those targets)
		for ( uint32_t i=0; i<targets.GetSize(); ++i )
		{
			result |= fBuild.Build( targets[ i ] );
		}
	}

	// even if the build has failed, we can still save the graph.
	// This is desireable because:
	// - it will save parsing the bff next time
	// - it will record the items that did build, so they won't build again
	fBuild.SaveDependencyGraph();

	return ( result == true ) ? FBUILD_OK : FBUILD_BUILD_UNKNOWN_TARGET;
}

// DisplayFBuildUsage
//------------------------------------------------------------------------------
void DisplayHelp()
{
	DisplayVersion();
	OUTPUT( "----------------------------------------------------------------------\n" );
	OUTPUT( "Usage: fbuild.exe [options] [target1]..[targetn]\n" );
	OUTPUT( "----------------------------------------------------------------------\n" );
	OUTPUT( "Options:\n" );
	OUTPUT( " -cache[read|write] Control use of the build cache.\n" );
	OUTPUT( " -clean	     Force a clean build.\n" );
	OUTPUT( " -config [path] Explicitly specify the config file to use\n" );
#ifdef DEBUG
	OUTPUT( " -debug         Break at startup, to attach debugger.\n" );
#endif
	OUTPUT( " -dist          Allow distributed compilation.\n" );
	OUTPUT( " -help          Should this help.\n" );
	OUTPUT( " -jX            Explicitly set worker thread count X, instead of\n" );
	OUTPUT( "                default of NUMBER_OF_PROCESSORS. Set to 0 to build\n" );
	OUTPUT( "                everything in the main thread.\n" );
	OUTPUT( " -nooutputbuffering Don't buffer output when writing to stdout and\n" );
	OUTPUT( "                stderr.  Useful if monitoring build output.\n" );
	OUTPUT( " -noprogress    Don't show the progress bar while building.\n" );
	OUTPUT( " -summary       Show a summary at the end of the build.\n" );
	OUTPUT( " -verbose       Show detailed diagnostic information. This will slow\n" );
	OUTPUT( "                down building.\n" );
	OUTPUT( " -version       Print version and exit. No other work will be\n" );
	OUTPUT( "                performed.\n" );
	OUTPUT( " -vs            VisualStudio mode. Same as '-noprogress'\n" );
	OUTPUT( "                and '-nooutputbuffering'.'\n" );
	OUTPUT( "----------------------------------------------------------------------\n" );
}

// DisplayVersion
//------------------------------------------------------------------------------
void DisplayVersion()
{
	#ifdef WIN64
		#define VERSION_PLATFORM "(x64)"
	#else
		#define VERSION_PLATFORM "(x86)"
	#endif
	#ifdef DEBUG
		OUTPUT( "FBuild - " VERSION_STRING " (DEBUG) " VERSION_PLATFORM "\n" );
	#else
		OUTPUT( "FBuild - " VERSION_STRING " " VERSION_PLATFORM "\n" );
	#endif
	#undef VERSION_PLATFORM
}

// CtrlHandler
//------------------------------------------------------------------------------
BOOL CtrlHandler( DWORD UNUSED( fdwCtrlType ) ) 
{
	// tell FBuild we want to stop the build cleanly
	FBuild::Get().SetStopBuild();

	// only printf output for the first break received
	static bool received = false;
	if ( received == false )
	{
		received = true;

		// get the console colours
		CONSOLE_SCREEN_BUFFER_INFO consoleInfo;
		VERIFY( GetConsoleScreenBufferInfo( GetStdHandle( STD_OUTPUT_HANDLE ), &consoleInfo ) );

		// print a big red msg
		VERIFY( SetConsoleTextAttribute( GetStdHandle( STD_OUTPUT_HANDLE ), FOREGROUND_RED ) );
		OUTPUT( "<<<< ABORT SIGNAL RECEIVED >>>>\n" );

		// put the console back to normal
		VERIFY( SetConsoleTextAttribute( GetStdHandle( STD_OUTPUT_HANDLE ), consoleInfo.wAttributes ) );
	}

	return TRUE; // tell Windows we've "handled" it
}

//------------------------------------------------------------------------------
