// UnitTest.h - interface for a unit test
//------------------------------------------------------------------------------
#pragma once
#ifndef TESTFRAMEWORK_UNITTEST_H
#define TESTFRAMEWORK_UNITTEST_H

#include "UnitTestManager.h"

//#include <Core/Env/Types.h>

// UnitTest - Tests derive from this interface
//------------------------------------------------------------------------------
class UnitTest
{
protected:
	explicit		UnitTest() { m_NextTestGroup = nullptr; }
	inline virtual ~UnitTest() {}

	virtual void RunTests() = 0;
	virtual const char * GetName() const = 0;
	virtual bool RunThisTestOnly() const { return false; }

private:
	friend class UnitTestManager;
	UnitTest * m_NextTestGroup;
};

// macros
//------------------------------------------------------------------------------
#define TEST_ASSERT( expression )									\
	do {															\
	__pragma(warning(push))											\
	__pragma(warning(disable:4127))									\
		if ( !( expression ) )										\
		{															\
			if ( UnitTestManager::AssertFailure(  #expression, __FILE__, __LINE__ ) ) \
			{														\
				__debugbreak();										\
			}														\
		}															\
	} while ( false );												\
	__pragma(warning(pop))

#define DECLARE_TESTS												\
	virtual void RunTests();										\
	virtual const char * GetName() const;

// use this macro when debugging to quickly restrict testing to one test
#define DECLARE_TESTS_RUN_ONLY_THIS_TEST							\
	DECLARE_TESTS													\
	virtual bool RunThisTestOnly() const { return true; }

#define REGISTER_TESTS_BEGIN( testGroupName )						\
	class RegisterTestsFor##testGroupName							\
	{																\
	public:															\
		RegisterTestsFor##testGroupName()							\
		{															\
			m_Test = new testGroupName;								\
			UnitTestManager::RegisterTestGroup( m_Test );			\
		}															\
		~RegisterTestsFor##testGroupName()							\
		{															\
			UnitTestManager::DeRegisterTestGroup( m_Test );			\
			delete m_Test;											\
		}															\
		testGroupName * m_Test;										\
	} s_RegisterTestsFor##testGroupName;							\
	const char * testGroupName::GetName() const						\
	{																\
		return #testGroupName;										\
	}																\
	void testGroupName::RunTests()									\
	{																\
		UnitTestManager & utm = UnitTestManager::Get();				\
		(void)utm;

#define REGISTER_TEST( testFunction )								\
		utm.TestBegin( #testFunction );								\
		testFunction();												\
		utm.TestEnd();

#define REGISTER_TESTS_END											\
	}

//------------------------------------------------------------------------------
#endif // TESTFRAMEWORK_UNITTEST_H