// Tracing
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "Tracing.h"

#include <stdio.h>
#include <stdarg.h>
#ifdef DEBUG
	#include <windows.h> // for OutputDebugStringA
#endif

// Static Data
//------------------------------------------------------------------------------
/*static*/ Tracing::Callback * Tracing::m_CallbackDebugSpam = nullptr;
/*static*/ Tracing::Callback * Tracing::m_CallbackOutput = nullptr;

#ifdef DEBUG
	// DebugSpam
	//------------------------------------------------------------------------------
	/*static*/ void Tracing::DebugSpam( const char * message )
	{
		// pass through callback if there is one
		if ( m_CallbackDebugSpam )
		{
			if ( (*m_CallbackDebugSpam)( message ) == false )
			{
				return; // callback wants msg supressed
			}
		}

		// normal output that goes to the TTY
		printf( message );

		// emit to the debugger as well if possible
		OutputDebugStringA( message );
	}

	// DebugSpamFormat
	//------------------------------------------------------------------------------
	/*static*/ void Tracing::DebugSpamFormat( const char * fmtString, ... )
	{
		const size_t BUFFER_SIZE( 8192 );
		char buffer[ BUFFER_SIZE ];

		va_list args;
		va_start(args, fmtString);
		vsnprintf_s( buffer, BUFFER_SIZE, _TRUNCATE, fmtString, args );
		va_end( args );

		DebugSpam( buffer );
	}

	// Warning
	//------------------------------------------------------------------------------
	/*static*/ void Tracing::Warning( const char * file, uint32_t line, const char * message )
	{
		// format a double clickable line
		const uint32_t BUFFER_SIZE( 8192 );
		char buffer[ BUFFER_SIZE ];
		sprintf_s( buffer, BUFFER_SIZE, "%s(%u): %s\n", file, line, message );

		// normal output that goes to the TTY
		puts( buffer );

		// emit to the debugger as well if possible
		#ifdef DEBUG
			OutputDebugStringA( buffer );
		#endif
	}

	// WarningFormat
	//------------------------------------------------------------------------------
	/*static*/ void Tracing::WarningFormat( const char * file, uint32_t line, const char * fmtString, ... )
	{
		const size_t BUFFER_SIZE( 8192 );
		char buffer[ BUFFER_SIZE ];

		va_list args;
		va_start(args, fmtString);
		vsnprintf_s( buffer, BUFFER_SIZE, _TRUNCATE, fmtString, args );
		va_end( args );

		Warning( file, line, buffer );
	}
#endif

// Output
//------------------------------------------------------------------------------
/*static*/ void Tracing::Output( const char * message )
{
	// pass through callback if there is one
	if ( m_CallbackOutput )
	{
		if ( (*m_CallbackOutput)( message ) == false )
		{
			return; // callback wants msg supressed
		}
	}

	// normal output that goes to the TTY
	printf( "%s", message );

	// emit to the debugger as well if possible
	#ifdef DEBUG
		OutputDebugStringA( message );
	#endif
}

// OutputFormat
//------------------------------------------------------------------------------
/*static*/ void Tracing::OutputFormat( const char * fmtString, ... )
{
	const size_t BUFFER_SIZE( 8192 );
	char buffer[ BUFFER_SIZE ];

	va_list args;
	va_start(args, fmtString);
	vsnprintf_s( buffer, BUFFER_SIZE, _TRUNCATE, fmtString, args );
	va_end( args );

	Output( buffer );
}

// Error
//------------------------------------------------------------------------------
/*static*/ void Tracing::FatalError( const char * message )
{
	// tty output
	puts( message );

	#ifdef DEBUG
		// to the debugger if available
		OutputDebugStringA( message );
	#endif

	// for now, we'll just break
	__debugbreak();
}

// ErrorFormat
//------------------------------------------------------------------------------
/*static*/ void Tracing::FatalErrorFormat( const char * fmtString, ... )
{
	const size_t BUFFER_SIZE( 8192 );
	char buffer[ BUFFER_SIZE ];

	va_list args;
	va_start(args, fmtString);
	vsnprintf_s( buffer, BUFFER_SIZE, _TRUNCATE, fmtString, args );
	va_end( args );

	FatalError( buffer );
}

// SetCallbackDebugSpam
//------------------------------------------------------------------------------
/*static*/ void Tracing::SetCallbackDebugSpam( Callback callback )
{
	m_CallbackDebugSpam = callback;
}

// SetCallbackOutput
//------------------------------------------------------------------------------
/*static*/ void Tracing::SetCallbackOutput( Callback callback )
{
	m_CallbackOutput = callback;
}

//------------------------------------------------------------------------------
