// FileIO.cpp
//------------------------------------------------------------------------------

// Includes
//------------------------------------------------------------------------------
#include "FileIO.h"
#include "FileStream.h"

#include <Core/Strings/AStackString.h>
#include <Core/Time/Timer.h>

#include <windows.h>

// Exists
//------------------------------------------------------------------------------
/*static*/ bool FileIO::FileExists( const char * fileName )
{
	// see if we can get attributes
	DWORD attributes = GetFileAttributes( fileName );
	if ( attributes == INVALID_FILE_ATTRIBUTES )
	{
		return false;
	}
	return true; // note this might not be file!
}

// Delete
//------------------------------------------------------------------------------
/*static*/ bool FileIO::FileDelete( const char * fileName )
{
	BOOL result = DeleteFile( fileName );
	if ( result == FALSE )
	{
		return false; // failed to delete
	}
	return true; // delete ok
}

// Copy
//------------------------------------------------------------------------------
/*static*/ bool FileIO::FileCopy( const char * srcFileName, const char * dstFileName,
							  bool allowOverwrite )
{
	BOOL failIfDestExists = ( allowOverwrite ? FALSE : TRUE );
	BOOL result = CopyFile( srcFileName, dstFileName, failIfDestExists );
	if ( result == FALSE )
	{
		// even if we allow overwrites, Windows will fail if the dest file
		// was read only, so we have to un-mark the read only status and try again
		if ( ( GetLastError() == ERROR_ACCESS_DENIED ) && ( allowOverwrite ) )
		{
			// see if dst file is read-only
			DWORD dwAttrs = GetFileAttributes( dstFileName );
			if ( dwAttrs == INVALID_FILE_ATTRIBUTES )
			{
				return false; // can't even get the attributes, nothing more we can do
			}
			if ( 0 == ( dwAttrs & FILE_ATTRIBUTE_READONLY ) ) 
			{ 
				return false; // file is not read only, so we don't know what the problem is
			}

			// try to remove read-only flag on dst file
			dwAttrs = ( dwAttrs & ~FILE_ATTRIBUTE_READONLY );
			if ( FALSE == SetFileAttributes( dstFileName, dwAttrs ) )
			{
				return false; // failed to remove read-only flag
			}

			// try to copy again
			result = CopyFile( srcFileName, dstFileName, failIfDestExists );
			return ( result == TRUE );
		}
	}

	return ( result == TRUE );
}

// FileMove
//------------------------------------------------------------------------------
/*static*/ bool FileIO::FileMove( const AString & srcFileName, const AString & dstFileName )
{
	return ( TRUE == ::MoveFile( srcFileName.Get(), dstFileName.Get() ) );
}

// GetFiles
//------------------------------------------------------------------------------
/*static*/ bool FileIO::GetFiles( const AString & path,
								  const AString & wildCard,
							      bool recurse,
								  Array< AString > * results )
{
	ASSERT( results );

	size_t oldSize = results->GetSize();
	if ( recurse )
	{
		// make a copy of the path as it will be modified during recursion
		AStackString< 256 > pathCopy( path );
		// ensure path ends with a backslash
		if ( pathCopy.EndsWith( '\\' ) == false )
		{
			pathCopy += '\\';
		}
		GetFilesRecurse( pathCopy, wildCard, results );
	}
	else
	{
		GetFilesNoRecurse( path.Get(), wildCard.Get(), results );
	}

	return ( results->GetSize() != oldSize );
}

//------------------------------------------------------------------------------
/*static*/ bool FileIO::GetCurrentDir( AString & output )
{
	char buffer[ MAX_PATH ];
	DWORD len = GetCurrentDirectory( MAX_PATH, buffer );
	if ( len != 0 )
	{
		output = buffer;
		return true;
	}
	return false;
}

//------------------------------------------------------------------------------
/*static*/ bool FileIO::GetTempDir( AString & output )
{
	char buffer[ MAX_PATH ];
	DWORD len = GetTempPath( MAX_PATH, buffer );
	if ( len != 0 )
	{
		output = buffer;
		return true;
	}
	return false;
}

// DirectoryCreate
//------------------------------------------------------------------------------
/*static*/ bool FileIO::DirectoryCreate( const AString & path )
{
	if ( CreateDirectory( path.Get(), nullptr ) )
	{
		return true;
	}

	// it failed - is it because it exists already?
	if ( GetLastError() == ERROR_ALREADY_EXISTS )
	{
		return true;
	}

	// failed, probably missing intermediate folders or an invalid name
	return false;
}

// DirectoryExists
//------------------------------------------------------------------------------
/*static*/ bool FileIO::DirectoryExists( const AString & path )
{
	DWORD res = GetFileAttributes( path.Get() );
	if ( ( res != INVALID_FILE_ATTRIBUTES ) &&
		 ( ( res & FILE_ATTRIBUTE_DIRECTORY ) != 0 ) )
	{
		return true; // exists and is a folder
	}
	return false; // doesn't exist, isn't a folder or some other problem
}

//------------------------------------------------------------------------------
/*static*/ bool FileIO::EnsurePathExists( const AString & path )
{
	// if the entire path already exists, nothing is to be done
	if( DirectoryExists( path ) )
	{
		return true;
	}

	// take a copy to locally manipulate
	AStackString<> pathCopy( path );
	pathCopy.Replace( '/', '\\' ); // make sure all slashes are as expected

	// make sure path has a tailing slash if not passed in with one
	// this ensures the logic below behaves the same for 'c:\xxx\' and 'c:\xxx' 
	if ( pathCopy.EndsWith( '\\' ) == false )
	{
		pathCopy += '\\';
	}

	char * slash = pathCopy.Find( '\\' );
	do
	{
		// truncate the string to the sub path
		*slash = '\000';
		if ( DirectoryExists( pathCopy ) == false )
		{
			// create this level
			if ( DirectoryCreate( pathCopy ) == false )
			{
				return false; // something went wrong
			}
		}
		*slash = '\\'; // put back the slash
		slash = pathCopy.Find( '\\', slash + 1 );
	}
	while ( slash );
	return true;
}

// CreateTempPath
//------------------------------------------------------------------------------
/*static*/ bool FileIO::CreateTempPath( const char * tempPrefix, AString & path )
{
	// get the system temp path
	char tempPath[ MAX_PATH ];
	DWORD len = GetTempPath( MAX_PATH, tempPath );
	if ( len == 0 )
	{
		return false;
	}	

	// create a temp file in the temp folder
	char tempFile[ MAX_PATH ];
	UINT uniqueVal = GetTempFileName( tempPath,		// LPCTSTR lpPathName
									  tempPrefix,	// LPCTSTR lpPrefixString
									  0,			// UINT uUnique,
									  tempFile );	// LPTSTR lpTempFileName
	if ( uniqueVal == 0 )
	{
		return false;
	}

	path = tempFile;
	return true;
}

// GetFileLastWriteTime
//------------------------------------------------------------------------------
/*static*/ uint64_t FileIO::GetFileLastWriteTime( const AString & fileName )
{
	WIN32_FILE_ATTRIBUTE_DATA fileAttribs; 
	if ( GetFileAttributesEx( fileName.Get(), GetFileExInfoStandard, &fileAttribs ) ) 
	{
		FILETIME ftWrite = fileAttribs.ftLastWriteTime; 
		uint64_t lastWriteTime = (uint64_t)ftWrite.dwLowDateTime | ( (uint64_t)ftWrite.dwHighDateTime << 32 ); 
		return lastWriteTime;
	} 
	return 0;
}

// SetFileLastWriteTime
//------------------------------------------------------------------------------
/*static*/ bool FileIO::SetFileLastWriteTime( const AString & fileName, uint64_t fileTime )
{
	// open the file
	// TOOD:B Check these args
	HANDLE hFile = CreateFile( fileName.Get(), GENERIC_WRITE, FILE_SHARE_WRITE, nullptr,
							OPEN_EXISTING, 0, nullptr);
	if( hFile == INVALID_HANDLE_VALUE )
	{
		return false;
	}

	// get the file time
	FILETIME ftWrite;
	ftWrite.dwLowDateTime = (uint32_t)( fileTime & 0x00000000FFFFFFFF );
	ftWrite.dwHighDateTime = (uint32_t)( ( fileTime & 0xFFFFFFFF00000000 ) >> 32 );
	if ( !SetFileTime( hFile, nullptr, nullptr, &ftWrite) ) // create, access, write
	{
		CloseHandle( hFile );
        return false;
	}

	// close the file
	CloseHandle( hFile );

	return true;
}

// SetReadOnly
//------------------------------------------------------------------------------
/*static*/ bool FileIO::SetReadOnly( const char * fileName, bool readOnly )
{
	// see if dst file is read-only
	DWORD dwAttrs = GetFileAttributes( fileName );
	if ( dwAttrs == INVALID_FILE_ATTRIBUTES )
	{
		return false; // can't even get the attributes, nothing more we can do
	}

	// determine the new attributes
	DWORD dwNewAttrs = ( readOnly ) ? ( dwAttrs | FILE_ATTRIBUTE_READONLY )
									: ( dwAttrs & ~FILE_ATTRIBUTE_READONLY );

	// nothing to do if they are the same
	if ( dwNewAttrs == dwAttrs )
	{
		return true;
	}

	// try to set change
	if ( FALSE == SetFileAttributes( fileName, dwNewAttrs ) )
	{
		return false; // failed
	}

	return true;
}

// GetFilesRecurse
//------------------------------------------------------------------------------
/*static*/ void FileIO::GetFilesRecurse( AString & pathCopy, 
										 const AString & wildCard,
										 Array< AString > * results )
{
	const uint32_t baseLength = pathCopy.GetLength();
	pathCopy += '*'; // don't want to use wildcard to filter folders

	// recurse into directories
	WIN32_FIND_DATA findData;
	HANDLE hFind = FindFirstFileEx( pathCopy.Get(), FindExInfoBasic, &findData, FindExSearchLimitToDirectories, nullptr, 0 );
	if ( hFind == INVALID_HANDLE_VALUE)
	{
		return;
	}

	do
	{
		if ( findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY )
		{
			// ignore magic '.' and '..' folders
			// (don't need to check length of name, as all names are at least 1 char
			// which means index 0 and 1 are valid to access)
			if ( findData.cFileName[ 0 ] == '.' &&
				( ( findData.cFileName[ 1 ] == '.' ) || ( findData.cFileName[ 1 ] == '\000' ) ) )
			{
				continue;
			}

			pathCopy.SetLength( baseLength );
			pathCopy += findData.cFileName;
			pathCopy += '\\';
			GetFilesRecurse( pathCopy, wildCard, results );
		}
	}
	while ( FindNextFile( hFind, &findData ) != 0 );
	FindClose( hFind );

	// do files in this directory
	pathCopy.SetLength( baseLength );
	pathCopy += wildCard;
	hFind = FindFirstFileEx( pathCopy.Get(), FindExInfoBasic, &findData, FindExSearchNameMatch, nullptr, 0 );
	if ( hFind == INVALID_HANDLE_VALUE)
	{
		return;
	}

	do
	{
		if ( findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY )
		{
			continue;
		}

		pathCopy.SetLength( baseLength );
		pathCopy += findData.cFileName;
		results->Append( pathCopy );
	}
	while ( FindNextFile( hFind, &findData ) != 0 );

	FindClose( hFind );
}

// GetFilesNoRecurse
//------------------------------------------------------------------------------
/*static*/ void FileIO::GetFilesNoRecurse( const char * path, 
										   const char * wildCard,
										   Array< AString > * results )
{
	AStackString< 256 > pathCopy( path );
	if ( pathCopy.EndsWith( '\\' ) == false )
	{
		pathCopy += '\\';
	}
	const uint32_t baseLength = pathCopy.GetLength();
	pathCopy += wildCard;

	WIN32_FIND_DATA findData;
	//HANDLE hFind = FindFirstFile( pathCopy.Get(), &findData );
	HANDLE hFind = FindFirstFileEx( pathCopy.Get(), FindExInfoBasic, &findData, FindExSearchNameMatch, nullptr, 0 );
	if ( hFind == INVALID_HANDLE_VALUE)
	{
		return;
	}

	do
	{
		if ( findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY )
		{
			continue;
		}

		pathCopy.SetLength( baseLength );
		pathCopy += findData.cFileName;
		results->Append( pathCopy );
	}
	while ( FindNextFile( hFind, &findData ) != 0 );

	FindClose( hFind );
}

// WorkAroundForWindowsFilePermissionProblem
//------------------------------------------------------------------------------
/*static*/ void FileIO::WorkAroundForWindowsFilePermissionProblem( const AString & fileName )
{
	// Sometimes after closing a file, subsequent operations on that file will
	// fail.  For example, trying to set the file time, or even another process
	// opening the file.
	//
	// This seems to be a known issue in windows, with multiple potential causes
	// like Virus scanners and possibly the behaviour of the kernel itself.
	//
	// A work-around for this problem is to attempt to open a file we just closed.
	// This will sometimes fail, but if we retry until it succeeds, we avoid the
	// problem on the subsequent operation.
	FileStream f;
	Timer timer;
	while ( f.Open( fileName.Get() ) == false )
	{
		Sleep( 1 );

		// timeout so we don't get stuck in here forever
		if ( timer.GetElapsed() > 1.0f )
		{
			ASSERT( false && "WorkAroundForWindowsFilePermissionProblem Failed!" );
			return;
		}
	}
	f.Close();
}

//------------------------------------------------------------------------------
